//
//  ZJTableViewItem.swift
//  NewRetail
//
//  Created by Javen on 2018/2/8.
//  Copyright © 2018年 . All rights reserved.
//

import UIKit

public typealias ZJTableViewItemBlock = (ZJTableViewItem) -> Void

open class ZJTableViewItem: NSObject {
    public weak var tableViewManager: ZJTableViewManager?
    public weak var section: ZJTableViewSection?
    public var cellIdentifier: String!
    /// cell 高度(如果要自动计算高度，使用 autoHeight(manager:) 方法，框架会算出高度，具体看 demo )
    /// 传 UITableViewAutomaticDimension 则是系统实时计算高度，可能会有卡顿、reload 弹跳等问题，不建议使用，有特殊需要可以选择使用
    public var cellHeight: CGFloat!
    /// cell 点击事件的回调
    public var selectionHandler: ZJTableViewItemBlock?
    public func setSelectionHandler<T: ZJTableViewItem>(_ handler: ((_ callBackItem: T) -> Void)?) {
        selectionHandler = { item in
            handler?(item as! T)
        }
    }

    public var deletionHandler: ZJTableViewItemBlock?
    public func setDeletionHandler<T: ZJTableViewItem>(_ handler: ((_ callBackItem: T) -> Void)?) {
        deletionHandler = { item in
            handler?(item as! T)
        }
    }

    public var labelText: String?
    public var detailLabelText: String?
    public var textAlignment: NSTextAlignment = .left
    public var detailTextAlignment: NSTextAlignment = .left
    public var image: UIImage?
    public var highlightedImage: UIImage?
    public var style: UITableViewCell.CellStyle = .default
    public var accessoryType: UITableViewCell.AccessoryType = .none
    public var selectionStyle: UITableViewCell.SelectionStyle = .default
    public var editingStyle: UITableViewCell.EditingStyle = .none
    public var accessoryView: UIView?
    public var isAutoDeselect: Bool = true
    public var isAutoHeight: Bool = false

    public var indexPath: IndexPath? {
        guard let section = tableViewManager?.sections.zj_indexOf(self.section) else { return nil}
        guard let rowIndex = self.section?.items.zj_indexOf(self) else { return nil}
        return IndexPath(item: rowIndex, section: section)
    }

    /// 尽量避免直接修改cell里面的元素，而是修改对应的item的属性，然后item.reload()来刷新cell。原因是直接修改cell没有修改修改数据源，TableView是重用的，容易出问题
    public var cell: UITableViewCell {
        guard let idxp = indexPath,
            let unwrappedCell = tableViewManager?.tableView.cellForRow(at: idxp) else { return UITableViewCell()}
        return unwrappedCell
    }

    public override init() {
        super.init()
        cellIdentifier = NSStringFromClass(type(of: self)).components(separatedBy: ".").last
        cellHeight = 44
    }

    public convenience init(title: String?) {
        self.init()
        labelText = title
    }
    
    public func reloadCell(_ animation: UITableView.RowAnimation = .automatic) {
        guard let idxp = indexPath else { return }
        tableViewManager?.reload(at: idxp, animation: animation)
    }
    
    public func reloadValue() {
        guard let idxp = indexPath else { return }
        tableViewManager?.reload(at: idxp, item: self)
    }

    public func reload() {
        reloadValue()
    }
    
    public func reloadTableView() {
        tableViewManager?.reload()
    }

    public func delete(_ animation: UITableView.RowAnimation = .automatic) {
        guard let idx = indexPath else { return }
        section?.items.remove(at: idx.row)
        tableViewManager?.tableView.deleteRows(at: [idx], with: animation)
    }

    /// 计算cell高度
    public func autoHeight(_ manager: ZJTableViewManager) {
        tableViewManager = manager
        let cell = manager.tableView.dequeueReusableCell(withIdentifier: cellIdentifier) as? ZJInternalCellProtocol
        if cell == nil {
            zj_log("please register cell")
        } else {
            cell?._item = self
            cellHeight = systemFittingHeightForConfiguratedCell(cell!)
        }
    }
    
    public func systemFittingHeightForConfiguratedCell(_ cell: ZJInternalCellProtocol) -> CGFloat {
        cell.cellWillAppear()
        let height = cell.systemLayoutSizeFitting(CGSize(width: tableViewManager?.tableView.frame.width ?? 0, height: 0), withHorizontalFittingPriority: .required, verticalFittingPriority: .fittingSizeLevel).height
        return height
    }
    
    
    public func _autoHeight() {
        guard isAutoHeight else { return }
        guard let cell = tableViewManager?.tableView.dequeueReusableCell(withIdentifier: cellIdentifier) as? ZJInternalCellProtocol else { return }
        cell._item = self
        cellHeight = _systemFittingHeightForConfiguratedCell(cell)
    }
    
    public func _systemFittingHeightForConfiguratedCell(_ cell: ZJInternalCellProtocol) -> CGFloat {
        let height = cell.systemLayoutSizeFitting(CGSize(width: tableViewManager?.tableView.frame.width ?? 0, height: 0), withHorizontalFittingPriority: .required, verticalFittingPriority: .fittingSizeLevel).height
        return height
    }
    
}
