//
//  ZJTableViewManager.swift
//  NewRetail
//
//  Created by Javen on 2018/2/8.
//  Copyright © 2018年 . All rights reserved.
//

import UIKit

func zj_log(_ item: Any, file: String = #file, line: Int = #line) {
    if ZJTableViewManager.isDebug {
        var logEntry: String = String()
        if let fileName = file.components(separatedBy: "/").last {
            logEntry.append("[\(fileName):\(line)] ")
        }
        print(logEntry + "\(item)")
    }
}

open class ZJTableViewManager: NSObject {
    public static var isDebug = false
    public weak var scrollDelegate: ZJTableViewScrollDelegate?
    public var tableView: UITableView!
    public var sections: [ZJTableViewSection] = []
    var defaultTableViewSectionHeight: CGFloat {
        return tableView.style == .grouped ? 0 : 0
    }

    public init(tableView: UITableView) {
        super.init()
        self.tableView = tableView
        tableView.delegate = self
        tableView.dataSource = self
    }

    /// use this method to update cell height after you change item.cellHeight.
    public func updateHeight() {
        tableView.beginUpdates()
        tableView.endUpdates()
    }

    public func register(_ nibClass: AnyClass, _ item: AnyClass, _ bundle: Bundle = Bundle.main) {
        if bundle.path(forResource: "\(nibClass)", ofType: "nib") != nil {
            tableView.register(UINib(nibName: "\(nibClass)", bundle: bundle), forCellReuseIdentifier: "\(item)")
        } else {
            tableView.register(nibClass, forCellReuseIdentifier: "\(item)")
        }
    }

    func sectionFrom(section: Int) -> ZJTableViewSection? {
        let section = sections.count > section ? sections[section] : nil
        return section
    }
    
    func optionalSectionFrom(section: Int) -> ZJTableViewSection? {
        let section = sections.count > section ? sections[section] : nil
        return section
    }

    func getSectionAndItem(indexPath: (section: Int, row: Int)) -> (section: ZJTableViewSection, item: ZJTableViewItem)? {
        guard let section = sectionFrom(section: indexPath.section) else { return nil}
        
        let item = section.items.count > indexPath.row ? section.items[indexPath.row] : nil
        guard let hasItem = item else { return nil}
        return (section, hasItem)
    }

    public func add(section: ZJTableViewSection) {
        section.tableViewManager = self
        sections.append(section)
    }

    public func remove(section: Any) {
        guard let zjSection = section as? ZJTableViewSection else { return }
        guard let index = sections.zj_indexOf(zjSection) else { return}
        sections.remove(at: index)
    }

    public func removeAllSections() {
        sections.removeAll()
    }

    public func reload() {
        tableView.reloadData()
    }
    
    public func reload(at indexPath: IndexPath, animation: UITableView.RowAnimation ) {
        tableView.reloadRows(at: [indexPath], with: animation)
    }
    
    public func reload(at indexPath: IndexPath, item: ZJTableViewItem ) {
        guard let unwrappedCell = tableView.cellForRow(at: indexPath) as? ZJInternalCellProtocol else { return }
        reload(unwrappedCell: unwrappedCell, item: item)
    }
    
    public func reload(unwrappedCell: ZJInternalCellProtocol, item: ZJTableViewItem ) {
        unwrappedCell.textLabel?.text = item.labelText
        unwrappedCell.textLabel?.textAlignment = item.textAlignment
        unwrappedCell.detailTextLabel?.text = item.detailLabelText
        unwrappedCell.detailTextLabel?.textAlignment = item.detailTextAlignment
        unwrappedCell.imageView?.image = item.image
        unwrappedCell.imageView?.highlightedImage = item.highlightedImage
        unwrappedCell._item = item
        unwrappedCell.cellWillAppear()
        unwrappedCell.selectionStyle = item.selectionStyle
        unwrappedCell.accessoryType = item.accessoryType
        // unwrappedCell.accessoryView = item.accessoryType == .none ? nil : item.accessoryView
    }
    
}

// MARK: - UITableViewDelegate

extension ZJTableViewManager: UITableViewDelegate {
    
    public func tableView(_: UITableView, willDisplay cell: UITableViewCell, forRowAt _: IndexPath) {
        guard let unwrappedCell = cell as? ZJInternalCellProtocol else { return }
        unwrappedCell.cellDidAppear()
    }

    public func tableView(_: UITableView, willDisplayHeaderView _: UIView, forSection section: Int) {
        guard let section = optionalSectionFrom(section: section) else { return }
        section.headerWillDisplayHandler?(section)
    }
    
    public func tableView(_: UITableView, willDisplayFooterView _: UIView, forSection section: Int) {
        guard let section = optionalSectionFrom(section: section) else { return }
        section.footerWillDisplayHandler?(section)
    }
    
    public func tableView(_: UITableView, didEndDisplaying cell: UITableViewCell, forRowAt _: IndexPath) {
        guard let unwrappedCell = cell as? ZJInternalCellProtocol else { return }
        unwrappedCell.cellDidDisappear()
    }

    public func tableView(_: UITableView, didEndDisplayingHeaderView _: UIView, forSection section: Int) {
        guard let section = optionalSectionFrom(section: section) else { return }
        section.headerDidEndDisplayHandler?(section)
    }
    
    public func tableView(_: UITableView, didEndDisplayingFooterView _: UIView, forSection section: Int) {
        guard let section = optionalSectionFrom(section: section) else { return }
        section.footerDidEndDisplayHandler?(section)
    }
    
    
    public func tableView(_ tableView: UITableView, heightForRowAt indexPath: IndexPath) -> CGFloat {
        let section = sections[indexPath.section]
        let item = section.items[indexPath.row]
        if item.isAutoHeight {
            return UITableView.automaticDimension
        }
        return item.cellHeight
    }
    
    public func tableView(_ tableView: UITableView, heightForHeaderInSection section: Int) -> CGFloat {
        guard let sectionItem = sectionFrom(section: section) else { return 0}
        if sectionItem.headerView != nil || (sectionItem.headerHeight > 0 && sectionItem.headerHeight != CGFloat.leastNormalMagnitude) {
            return sectionItem.headerHeight
        }

        if let title = sectionItem.headerTitle {
            let label = UILabel(frame: CGRect(x: 0, y: 0, width: tableView.frame.width - 40, height: CGFloat.greatestFiniteMagnitude))
            label.text = title
            label.font = UIFont.preferredFont(forTextStyle: .footnote)
            label.sizeToFit()
            return label.frame.height + 20.0
        } else {
            return defaultTableViewSectionHeight
        }
    }

    public func tableView(_: UITableView, heightForFooterInSection section: Int) -> CGFloat {
        guard let sectionItem = sectionFrom(section: section) else { return 0}
        return sectionItem.footerHeight
    }
    
    public func tableView(_: UITableView, viewForHeaderInSection section: Int) -> UIView? {
        guard let sectionItem = sectionFrom(section: section) else { return nil}
        return sectionItem.headerView
    }

    public func tableView(_: UITableView, viewForFooterInSection section: Int) -> UIView? {
        guard let sectionItem = sectionFrom(section: section) else { return nil}
        return sectionItem.footerView
    }

    
    
    public func tableView(_ tableView: UITableView, didSelectRowAt indexPath: IndexPath) {
        guard let tuples = getSectionAndItem(indexPath: (indexPath.section, indexPath.row)) else { return }
        if tuples.item.isAutoDeselect {
            tableView.deselectRow(at: indexPath, animated: true)
        }
        tuples.item.selectionHandler?(tuples.item)
    }

    public func tableView(_: UITableView, editingStyleForRowAt indexPath: IndexPath) -> UITableViewCell.EditingStyle {
        guard let tuples = getSectionAndItem(indexPath: (indexPath.section, indexPath.row)) else { return .none }
        return tuples.item.editingStyle
    }
}

// MARK: - UITableViewDataSource

extension ZJTableViewManager: UITableViewDataSource {
   
    public func tableView(_: UITableView, numberOfRowsInSection section: Int) -> Int {
        guard let sectionItem = sectionFrom(section: section) else { return 0}
        return sectionItem.items.count
    }

    public func tableView(_ tableView: UITableView, cellForRowAt indexPath: IndexPath) -> UITableViewCell {
        guard let tuples = getSectionAndItem(indexPath: (indexPath.section, indexPath.row)) else { return UITableViewCell() }
        let item = tuples.item
        item.tableViewManager = self
        var cell = tableView.dequeueReusableCell(withIdentifier: item.cellIdentifier) as? ZJInternalCellProtocol
        if cell == nil {
            cell = (ZJDefaultCell(style: item.style, reuseIdentifier: item.cellIdentifier) as ZJInternalCellProtocol)
        }
        
        let unwrappedCell = cell!
        reload(unwrappedCell: unwrappedCell, item: item)
        
        return unwrappedCell
    }
    
    public func numberOfSections(in _: UITableView) -> Int {
        return sections.count
    }

    public func tableView(_: UITableView, titleForHeaderInSection section: Int) -> String? {
        guard let sectionItem = sectionFrom(section: section) else { return nil}
        return sectionItem.headerTitle
    }

    public func tableView(_: UITableView, titleForFooterInSection section: Int) -> String? {
        guard let sectionItem = sectionFrom(section: section) else { return nil}
        return sectionItem.footerTitle
    }
    
    public func tableView(_: UITableView, commit editingStyle: UITableViewCell.EditingStyle, forRowAt indexPath: IndexPath) {
        guard let tuples = getSectionAndItem(indexPath: (indexPath.section, indexPath.row)) else { return }
        if editingStyle == .delete {
            if let handler = tuples.item.deletionHandler {
                handler(tuples.item)
            }
        }
    }
}

extension Array where Element: Equatable {
    func zj_indexOf(_ element: Element?) -> Int? {
        guard let hasElement = element else { return nil}
        var index: Int?
        #if swift(>=5)
            index = firstIndex { (e) -> Bool in
                e == hasElement
            }
        #else
            index = self.index(where: { (e) -> Bool in
                e == hasElement
            })
        #endif
        return index
    }
}
