//
//  YFDataManager.swift
//  YFitkit
//
//  Created by Mac on 2020/5/16.
//

import Foundation
import RxSwift
import YFitSDK
import NEUtils

// MARK: - 数据类型
public enum YFDataType {
    case hybridInfo
    case otadevice
    case notify(YFNotifyType)
    case record(YFRecordType, String)
}

// MARK: - SRDataManager
public class YFDataManager {
    
    // MARK: - field
    var hybridInfo: YFHybridInfo = YFHybridInfo.readForCache()
    ///: - Private
    private let updateSubject = PublishSubject<YFDataType>()
    private let disposeBag = DisposeBag()
    private let otaDevice: YFOtaDevice = YFOtaDevice()

    init() {
        _ = NEWeatherTask.shared
        observes()
    }
    
    // MARK: - observes
    private func observes() {
        NEAppNotifyUtils.observable.subscribe(onNext: {[weak self] type in
            guard let strong = self else {return }
            switch type {
            case .willResignActive: if YFKitService.isConnected { YFCommand.appSensor(.off).writeToDevice() }
            case .didBecomeActive: if YFKitService.isConnected { strong.syncDatas() }
            default: break
            }
        }).disposed(by: disposeBag)
        
        
        YFKitService.productStatusOutput.subscribe(onNext: {[weak self] status in
            guard let strong = self else {return }
            strong.productStatusChanged(status)
        }).disposed(by: disposeBag)
        
        YFKitService.receiveDatasOutput.subscribe(onNext: {[weak self] dictionary in
            guard let strong = self else {return }
            strong.receiveDatas(dictionary)
        }).disposed(by: disposeBag)
        
        weatherObserves()
    }
    
    private func productStatusChanged(_ status: YFProductStatus) {
        switch status {
        case .completed:
            DispatchQueue.main.async { YFHealthStore.requestAuthorization() }
            NEWeatherTask.lastPushTime = 0
            YFCommand.pair().writeToDevice { [weak self] (error) in
                guard let strong = self, error == nil else { return }
                DispatchQueue.main.async { strong.syncDatas() }
            }
        case .powerOff, .disconnected:
            weatherPause()
        default: break
        }
    }
}

///: - 天气同步
extension YFDataManager {
    
    private func weatherStart() {
        guard YFKitService.isSW294 else { return }
        NEWeatherTask.start()
    }
    
    private func weatherPause() {
        NEWeatherTask.pause()

    }
    
    private func weatherObserves() {
        NEWeatherTask.updatedPushOutput.subscribe(onNext: {[weak self] (_) in
            guard let strong = self else {return }
            strong.toPushWeaterInfo()
        }).disposed(by: disposeBag)
    }
    
    private func toPushWeaterInfo() {
        guard YFKitService.binded else { return }
        guard YFKitService.isSW294, YFKitService.isConnected,
              NEWeatherInfo.hourItems != nil else { NEWeatherTask.isActive = false; return }
        for i in 0...3 {
            if let command = YFCommand.weather(i) { command.writeToDevice() }
            if i == 3 {
                NEWeatherTask.lastPushTime = Date().timeIntervalSince1970
                NEWeatherTask.isActive = false
            }
        }
    }
}


///: - 数据同步与接收
extension YFDataManager {
    
    // MARK: - 同步数据
    private func syncDatas() {
        /// 开启监听
        YFCommand.appSensor(.on).writeToDevice()
        /// 同步时间
        YFCommand.syncTime().writeToDevice()
        /// 同步语言
        YFCommand.syncLanguage().writeToDevice()
        /// 同步天气
        weatherStart()
        /// 读取OTA， 用于判断固件升级条件
        readOTAInfo()
        
        /// ----- 这两项首次默认设计开启 ----
        /// 只设置一次
        if YFDataManager.addUUID() {
            /// 设置短信提醒
            YFCommand.setMessageNotify(YFMessageNotify()).writeToDevice()
            /// 设置通知应用类型开关
            YFCommand.setMessageNotifyType(YFMessageNotifyType()).writeToDevice()
        }
        
        
        /// 由于设计了同步参数， 需要延时读取
        DispatchQueue.main.asyncAfter(deadline: .now()+0.5) {
            //// 读取用户信息
            YFCommand.readUserInfo().writeToDevice()
            /// 读取目标信息
            YFCommand.readGoalInfo().writeToDevice()
            /// 读取混杂混杂数据
            YFCommand.readHybrid().writeToDevice()
            /// 如果运行在后台；同步完数据要关闭传感器
            if NELocationManager.runningInBackground() {
                YFCommand.appSensor(.off).writeToDevice()
            }
        }        
    }
    
    /// 读取OTA， 用于判断固件升级条件
    private func readOTAInfo() {
        /// 读取OTA设备信息
        YFCommand.readOTADeviceInfo().writeToDevice()
        /// 读取OTA状态信息
        YFCommand.readOTAStatus().writeToDevice()
    }
    
    
    // MARK: - 接收到手环数据 处理
    private func receiveDatas(_ dict: Dictionary<String, Any>) {
        YFLog.d("Received: \(dict)")
        guard let dataType: Int = dict["DataType"] as? Int,
              let type = K6_DataFuncType(rawValue: dataType) else { return }
        let data = dict["Data"]
        
        /// 混杂数据 - 多项
        if type == .DATA_TYPE_DEV_SYNC{
            guard let items = data as? [Dictionary<String, Any>] else { return }
            hybridInfo.update(forItems: items)
            updateSubject.onNext(.hybridInfo)
        }
        /// 记录数据
        else if type.isRecord {
            guard let type = YFRecordType.parse(type),
                  let dateStr = type.parse(dict) else { return }
            updateSubject.onNext(.record(type, dateStr))
        }
        
        
        guard let dict = data as? Dictionary<String, Any> else { return }
        /// 混杂数据 - 单项
        if type.isHybrid {
            hybridInfo.update(for: type, data: dict)
        }
        /// OTA 设备
        else if type.isOta {
            otaDevice.update(forItem: dict)
            updateSubject.onNext(.otadevice)
        }
        /// 通知事件
        else if type.isEvent {
            guard let type = YFNotifyType.parse(type, dict) else { return }
            updateSubject.onNext(.notify(type))
        }
    }
}


/// : - 记录已连接过的设备
/// 方便作为设备首次连入时，对设备默认初始值设置
extension YFDataManager {
    private static let uuidsKey = "YFituuidsKey"
    private static var _uuids: [String]?
    private static var uuids: [String] {
        get {
            if let array = _uuids { return array}
            guard let array = UserDefaults.standard.array(forKey: uuidsKey) as? [String] else { _uuids = []; return [] }
            _uuids = array
            return array
        }
        set {
            _uuids = newValue
            UserDefaults.standard.set(newValue, forKey: uuidsKey)
            UserDefaults.standard.synchronize()
        }
    }
    
    /// : - 添加 uuid
    /// return： - 返回 false,  表示已添加;  true 添加成功
    static func addUUID() -> Bool {
        guard let uuid: String = YFKitService.currentPeripheral?.uuid else { return false }
        if uuids.contains(uuid) { return false }
        uuids.append(uuid)
        return true
    }
    
    /// ：- 移除 uuid
    /// return： - 返回 false,  表示没有当前设备;  true 删除成功
    static func removeUUID() -> Bool {
        guard let uuid: String = YFKitService.currentPeripheral?.uuid else { return false }
        guard let indx = uuids.firstIndex(of: uuid) else { return false }
        uuids.remove(at: indx)
        return true
    }
}

/// : - 清除方法
extension YFDataManager {
    
    ///: - 清除数据信息
    private static func clearInfo() {
        shared.hybridInfo = YFHybridInfo()
    }
    
    ///: - 清除已读设置记录标志
    private static func clearReaded() {
        YFHybridInfo.readed = false
        YFClockInfo.readed = false
    }
    
    
    ///: - 删除用户数据时调用
    private static func _deletePersonalData() {
        YFRecordTotal.remove()
        YFRecordHeart.remove()
        YFRecordHeartExercise.remove()
        YFRecordSleep.remove()
        YFRecordExercise.remove()
    }
    
}
    


//MARK: - Class methods
extension YFDataManager {
    /// 单例
    static let shared: YFDataManager = YFDataManager()
    
    public static var hybridInfo: YFHybridInfo { shared.hybridInfo }

    /// : - Subject - Public output
    public static var updateOutput: Observable<YFDataType> {
        return shared.updateSubject.asObservable()
    }
    
    /// : - 删除个人数据
    public static func deletePersonalData() { _deletePersonalData() }
    
    ///: -  后台挂起唤醒要执行的操作
    public static func performWakeUp() {
        guard NELocationManager.runningInBackground() else { return }
        if YFKitService.isConnected {
            shared.syncDatas()
        } else {
            
        }
    }
}
