//
//  YFHealthStore.swift
//  YFitKit
//
//  Created by Mac on 2021/11/10.
//

import Foundation

public typealias HealthStoreClosure = (Bool, Error?) -> Void

private let kCmdOpQueue = DispatchQueue(label: "com.jpaxh.yfkit.HealthStoreQueue")

func healthStoreSyncAction(_ block: @escaping () -> Void) {
    kCmdOpQueue.sync {
        block()
    }
}

func healthStoreASyncAction(dalay: TimeInterval, _ block: @escaping () -> Void) {
    kCmdOpQueue.asyncAfter(deadline: .now() + dalay, execute: {
        block()
    })
}

enum YFHealthError: Error {
    case unavailable
}

#if HEALTHEANBLE

import HealthKit

enum YFHealthSampleType {
    case step, calories, distance, heart, sleep
    var sampleType: HKSampleType {
        switch self {
        case .step:     return HKSampleType.quantityType(forIdentifier: .stepCount)!
        case .calories: return HKSampleType.quantityType(forIdentifier: .activeEnergyBurned)!
        case .distance: return HKSampleType.quantityType(forIdentifier: .distanceWalkingRunning)!
        case .heart:    return HKSampleType.quantityType(forIdentifier: .heartRate)!
        case .sleep:    return HKSampleType.categoryType(forIdentifier: .sleepAnalysis)!
        }
    }
    
    func quantity(of value: Double) -> HKQuantity? {
        guard value > 0 else { return nil }
        switch self {
        case .step:     return HKQuantity(unit: HKUnit.count(), doubleValue: value)
        case .calories: return HKQuantity(unit: HKUnit.kilocalorie(), doubleValue: value/1000)
        case .distance:
            if value > 10 {
                return HKQuantity(unit: HKUnit.meterUnit(with: .kilo), doubleValue: value/1000)
            } else {
                return HKQuantity(unit: HKUnit.meterUnit(with: .deca), doubleValue: value/10)
            }
        case .heart:    return HKQuantity(unit: HKUnit.count().unitDivided(by: HKUnit.minute()), doubleValue: value)
        case .sleep:    return nil
        }
    }
    
    static var shareTypes: Set<HKSampleType> = {
        let collection = [step, calories, distance, heart, sleep]
        return Set(collection.map { $0.sampleType })
    } ()
}

public class YFHealthStore {
            
    private var available: Bool { HKHealthStore.isHealthDataAvailable() }

    private lazy var store: HKHealthStore  = HKHealthStore()
    
    private var authorEnable: Bool = false
    
    private init() { }
    
    /// 谓词过滤：指定日期
    private func predicate(greather time: TimeInterval) -> NSPredicate {
        let startDate = Date(timeIntervalSince1970: time)
        let endDate = YFCalendarUtils.calendar.date(byAdding: .year, value: 100, to: startDate)
        return HKQuery.predicateForSamples(withStart: startDate, end: endDate, options: .strictStartDate)
    }
    
   
    /// 移除上次保存的数据
    private func remove(greather time: TimeInterval, of type: YFHealthSampleType, closure: HealthStoreClosure? = nil) {
        let predicate: NSPredicate = predicate(greather: time)
        store.deleteObjects(of: type.sampleType, predicate: predicate) { (success, count, error) in
            closure?(success, error)
        }
    }
    
    /// Step \ Calories \ Distance \ HeartRate
    private func quantitySample(of type: YFHealthSampleType, with value: Double,
                                from start: Date, to end: Date) -> HKQuantitySample? {
        let calendar = YFCalendarUtils.calendar
        let endDate = calendar.date(byAdding: .second, value: -1, to: end)!
        if let quantityType = type.sampleType as? HKQuantityType, let quantity = type.quantity(of: value) {
            return HKQuantitySample(type: quantityType, quantity: quantity, start: start, end: endDate)
        }
        return nil
    }
    
    /// Sleep
    private func categorySample(sleep type: YFSleepType,
                                from start: Date, to end: Date) -> HKCategorySample? {
        
        guard type != .none else { return nil}
        guard type != .wakeup else { return nil}
        if let categoryType = YFHealthSampleType.sleep.sampleType as? HKCategoryType {
            var value: HKCategoryValueSleepAnalysis = .inBed
            switch type {
            case .light, .deep: value = .asleep
            default: if #available(iOS 10.0, *) { value = .awake }
            }
            return HKCategorySample(type: categoryType, value: value.rawValue, start: start, end: end)
        }
        return nil
    }
}

extension YFHealthStore {
    
    /// 请求授权
    private func requestAuthorization(closure: HealthStoreClosure? = nil) {
        guard available else { closure?(false, YFHealthError.unavailable); return }
        store.requestAuthorization(toShare: YFHealthSampleType.shareTypes, read: nil, completion: { [weak self] (success, error) in
            guard let strong = self else { closure?(false, YFHealthError.unavailable); return }
            strong.authorEnable = success
            closure?(success, error)
        })
    }

    /// 存储运动数据 — 步数 \ 卡路里 \ 距离（24小时）
    private func save(sport model: YFTotal, closure: HealthStoreClosure? = nil) {
        guard authorEnable else { closure?(false, YFHealthError.unavailable); return }
        var _: Bool = false

        let dayTime = YFCalendarUtils.dayDate(for: Date(timeIntervalSince1970: model.startSecs)).timeIntervalSince1970
        let array = YFTotal.query(start: dayTime)
        let startSecs = dayTime

        let intervalArray = YFSportInterval.parse(with: array)
        guard intervalArray.first != nil else { closure?(true, nil); return }
        
        var sampleArray: [HKQuantitySample] = []
        for m in intervalArray {
            let startDate = Date(timeIntervalSince1970: m.startSecs)
            let endDate = Date(timeIntervalSince1970: m.startSecs + 1)
            if m.walkSteps > 0,
               let sample = self.quantitySample(of: .step, with: Double(m.walkSteps), from: startDate, to: endDate) {
                sampleArray.append(sample)
            }
            if m.walkDistance > 0,
               let sample = self.quantitySample(of: .distance, with: Double(m.walkDistance), from: startDate, to: endDate) {
                sampleArray.append(sample)
            }
            if m.walkCalories > 0,
               let sample = self.quantitySample(of: .calories, with: Double(m.walkCalories), from: startDate, to: endDate) {
                sampleArray.append(sample)
            }
        }
        
        
        remove(greather: startSecs, of: .step)
        remove(greather: startSecs, of: .calories)
        remove(greather: startSecs, of: .distance) { [weak self] (success, error) in
            guard let strong = self, success else { closure?(false, YFHealthError.unavailable); return }
            strong.store.save(sampleArray, withCompletion: { (success, error) in
                closure?(success, error)
            })
        }
    }
    
    
    /// 存储睡眠
    private func save(sleep _array: [YFSleep], closure: HealthStoreClosure? = nil) {
        guard authorEnable else { closure?(false, YFHealthError.unavailable); return }
        let array = YFSleepDuration.parse(with: _array)
        guard let first =  array.first else { closure?(true, nil); return }
        var sampleArray: [HKCategorySample] = []
        for m in array {
            let startDate = Date(timeIntervalSince1970: m.startTime)
            let endDate = Date(timeIntervalSince1970: m.endTime)
            if let sample = self.categorySample(sleep: m.type, from: startDate, to: endDate) {
                sampleArray.append(sample)
            }
        }
        
        remove(greather: first.startTime, of: .sleep) { [weak self] (success, error) in
            guard let strong = self, success else { closure?(false, YFHealthError.unavailable); return }
            strong.store.save(sampleArray, withCompletion: { (success, error) in
                closure?(success, error)
            })
        }
    }
    
    /// 存储静态心率
    private func save(heart array: [YFHeart], closure: HealthStoreClosure? = nil) {
        guard authorEnable else { closure?(false, YFHealthError.unavailable); return }
        guard let first =  array.first else { closure?(true, nil); return }

        var sampleArray: [HKQuantitySample] = []
        let calendar = YFCalendarUtils.calendar
        for m in array {
            let startDate = Date(timeIntervalSince1970: m.time)
            let endDate = calendar.date(byAdding: .minute, value: 1, to: startDate)!
            if let sample = self.quantitySample(of: .heart, with: Double(m.heartNum), from: startDate, to: endDate) {
                sampleArray.append(sample)
            }
        }
        
        remove(greather: first.time, of: .heart) { [weak self] (success, error) in
            guard let strong = self, success else { closure?(false, YFHealthError.unavailable); return }
            strong.store.save(sampleArray, withCompletion: { (success, error) in
                closure?(success, error)
            })
        }
    }

}

#else

public class YFHealthStore {
    private init() { }
    private func requestAuthorization(closure: HealthStoreClosure? = nil) { closure?(false, YFHealthError.unavailable) }
    private func save(sport model: YFTotal, closure: HealthStoreClosure? = nil) { closure?(false, YFHealthError.unavailable) }
    private func save(sleep array: [YFSleep], closure: HealthStoreClosure? = nil) { closure?(false, YFHealthError.unavailable) }
    private func save(heart array: [YFHeart], closure: HealthStoreClosure? = nil) { closure?(false, YFHealthError.unavailable) }
}

#endif

extension YFHealthStore {

    private static let shared = YFHealthStore()

    static func requestAuthorization(closure: HealthStoreClosure? = nil) {
        shared.requestAuthorization(closure: closure)
    }

    /// 存储运动数据 — 步数 \ 卡路里 \ 距离（24小时）
    static func save(sport array: [YFTotal], closure: HealthStoreClosure? = nil) {
        guard let first = array.first else { closure?(true, nil); return }
        shared.save(sport: first, closure: closure)
    }
    
    /// 存储随眠
    static func save(sleep array: [YFSleep], closure: HealthStoreClosure? = nil) {
        shared.save(sleep: array, closure: closure)
    }
    
    /// 存储静态心率 - 144
    static func save(heart array: [YFHeart], closure: HealthStoreClosure? = nil) {
        shared.save(heart: array, closure: closure)
    }
}
