//
//  YFHybridInfo.swift
//  YFitKit
//
//  Created by Mac on 2021/4/12.
//

import Foundation
import HandyJSON
import YFitSDK

/// 设置信息、设备信息  等混杂参数
public class YFHybridInfo: YFModelBase {
    ///: -
    /// 电池
    public var batteryInfo: YFBattery = YFBattery()
    /// 设备信息
    public var deviceInfo: YFFirmware = YFFirmware()
    
    ///: -
    /// 单位
    public var unitInfo: YFUnit = YFUnit()
    /// 时间制
    public var hourlyInfo: YFHourly = YFHourly()
    /// 语言
    public var languageInfo: YFLanguage = YFLanguage()
    
    ///: -
    /// 用户信息
    public var userInfo: YFUserInfo = YFUserInfo()
    /// 闹钟信息
    public var alarmInfo: YFClockInfo = YFClockInfo()
    /// 目标信息
    public var goalInfo: YFGoalInfo = YFGoalInfo()

    ///: -
    /// 勿扰时段
    public var disturbInfo: YFDisturb = YFDisturb()
    /// 心率监测
    public var heartRateMonitor: YFHeartRateMonitor = YFHeartRateMonitor()
    /// 抬腕亮屏
    public var remindRaiseLight: YFRaiseLight = YFRaiseLight()
    /// 短信通知
    public var messageNotifyInfo: YFMessageNotify = YFMessageNotify()
    /// 通知提醒应用类型开关
    public var remindNotifySwitch: YFMessageNotifyType = YFMessageNotifyType()

    ///: -
    /// 来电提醒
    public var remindCall: YFRemindCall = YFRemindCall()
    /// 达标提醒
    public var remindTarget: YFRemindTarget = YFRemindTarget()
    /// 久坐提醒
    public var remindSedentary: YFRemindSedentary = YFRemindSedentary()
    /// 喝水提醒
    public var remindDrink: YFRemindDrink = YFRemindDrink()
   
}

// MARK: - CSV

extension YFHybridInfo {
    func csvContent(_ userName: String) -> String {
        var content = batteryInfo.csvContent()
        content += deviceInfo.csvContent()
        content += unitInfo.csvContent()
        content += hourlyInfo.csvContent()
        content += languageInfo.csvContent()

        content += userInfo.csvContent(nickname: userName)
        content += goalInfo.csvContent()
        
        content += heartRateMonitor.csvContent()
        content += remindRaiseLight.csvContent()
        content += messageNotifyInfo.csvContent()
        content += remindNotifySwitch.csvContent()

        content += remindCall.csvContent()
        content += remindTarget.csvContent()
        content += remindSedentary.csvContent()
        content += remindDrink.csvContent()

        return content
    }
    
    public static func csvContent(_ userName: String = "") -> String {
        return YFDataManager.hybridInfo.csvContent(userName)
    }
}


// MARK: - 数据缓存
extension YFHybridInfo {

    func saveToCache() {
        YFHybridInfo.default = self
    }
    
    static func readForCache() -> YFHybridInfo {
        if let cache = YFHybridInfo.default { return cache }
        return YFHybridInfo()
    }
    
    private static let key = "YFHybridInfoBandKitKey"
    private static var _default: YFHybridInfo?
    private static var `default`: YFHybridInfo? {
        get {
            if let info = _default { return info}
            guard let jsonString = UserDefaults.standard.string(forKey: key) else { return nil }
            guard let info = YFHybridInfo.deserialize(from: jsonString) else { return nil }
            _default = info
            return _default
        }
        set {
            _default = newValue
            if _default != nil {
                UserDefaults.standard.set(_default!.jsonString, forKey: key)
            } else {
                UserDefaults.standard.removeObject(forKey: key)
            }
            UserDefaults.standard.synchronize()
        }
    }
}

/// 新连接设备是否读取过数据
extension YFHybridInfo {
    private static let readedKey = "YFHybridInfoReadedKey"
    public static var readed: Bool {
        get {
            UserDefaults.standard.bool(forKey: readedKey)
        }
        set {
            UserDefaults.standard.set(newValue, forKey: readedKey)
            UserDefaults.standard.synchronize()
        }
    }
}

///: - 接收到设备数据来更新
extension YFHybridInfo {
    func update(forItems items: [Dictionary<String, Any>]) {
        let types = items.map { _update(for: $0) }
        saveToCache()
        YFHybridInfo.readed = true
        if types.contains(.DATA_TYPE_ALARM) { YFClockInfo.readed = true }
    }
    
    func update(for type: K6_DataFuncType, data: Dictionary<String, Any>) {
        _update(for: type, data: data)
        saveToCache()
        if type == .DATA_TYPE_ALARM { YFClockInfo.readed = true }
    }
    
    @discardableResult
    private func _update(for item: Dictionary<String, Any>) -> K6_DataFuncType? {
        guard let dataType: Int = item["DataType"] as? Int,
              let data: Dictionary<String, Any> = item["Data"] as? Dictionary<String, Any>,
              let type = K6_DataFuncType(rawValue: dataType) else { return nil }
        _update(for: type, data: data)
        return type
    }
    
    private func _update(for type: K6_DataFuncType, data: Dictionary<String, Any>) {
        switch type {
        ///: -
        case .DATA_TYPE_BATTERY_INFO:
            guard let model = YFBattery.deserialize(from: data) else { return }
            self.batteryInfo = model
        case .DATA_TYPE_DEVINFO:
            guard let model = YFFirmware.deserialize(from: data) else { return }
            self.deviceInfo = model
        
        ///: -
        case .DATA_TYPE_UNIT_SETTING:
            guard let model = YFUnit.deserialize(from: data) else { return }
            self.unitInfo = model
        case .DATA_TYPE_TIME:
            guard let model = YFHourly.deserialize(from: data) else { return }
            self.hourlyInfo = model
        case .DATA_TYPE_LANGUAGE_SETTING:
            guard let model = YFLanguage.deserialize(from: data) else { return }
            self.languageInfo = model
         
        ///: -
        case .DATA_TYPE_USERINFO:
            guard let model = YFUserInfo.deserialize(from: data) else { return }
            self.userInfo.update(forDevice: model)
        case .DATA_TYPE_ALARM:
            guard let model = YFClockInfo.deserialize(from: data) else { return }
            self.alarmInfo = model
        case .DATA_TYPE_SET_TARGET:
            guard let model = YFGoalInfo.deserialize(from: data) else { return }
            self.goalInfo = model
            
        ///: -
        case .DATA_TYPE_FORGET_DISTURB:
            guard let model = YFDisturb.deserialize(from: data) else { return }
            self.disturbInfo = model
        case .DATA_TYPE_HEART_AUTO_SWITCH:
            guard let model = YFHeartRateMonitor.deserialize(from: data) else { return }
            self.heartRateMonitor = model
        case .DATA_TYPE_HAND_RISE_SWITCH:
            guard let model = YFRaiseLight.deserialize(from: data) else { return }
            self.remindRaiseLight = model
        case .DATA_TYPE_MESSAGE_ALARM:
            guard let model = YFMessageNotify.deserialize(from: data) else { return }
            self.messageNotifyInfo = model
        case .DATA_TYPE_MESSAGE_SWITCH:
            guard let model = YFMessageNotifyType.deserialize(from: data) else { return }
            self.remindNotifySwitch = model
            
        ///: -
        case .DATA_TYPE_CALL_ALARM:
            guard let model = YFRemindCall.deserialize(from: data) else { return }
            self.remindCall = model
        case .DATA_TYPE_TARGET_ALARM:
            guard let model = YFRemindTarget.deserialize(from: data) else { return }
            self.remindTarget = model
        case .DATA_TYPE_SITTING_REMIND:
            guard let model = YFRemindSedentary.deserialize(from: data) else { return }
            self.remindSedentary = model
        case .DATA_TYPE_DRINK_ALARM:
            guard let model = YFRemindDrink.deserialize(from: data) else { return }
            self.remindDrink = model
        
        default: break
        }
    }
}


extension K6_DataFuncType {
    
    var isHybrid:Bool { K6_DataFuncType.hybridCollection.contains(self) }
    
    static let hybridCollection: [K6_DataFuncType] = [DATA_TYPE_BATTERY_INFO,
                                                      DATA_TYPE_DEVINFO,
    
                                                      DATA_TYPE_UNIT_SETTING,
                                                      DATA_TYPE_HEART_AUTO_SWITCH,
                                                      DATA_TYPE_LANGUAGE_SETTING,
                                                      
                                                      DATA_TYPE_USERINFO,
                                                      DATA_TYPE_ALARM,
                                                      DATA_TYPE_SET_TARGET,
                                                      
                                                      DATA_TYPE_FORGET_DISTURB,
                                                      DATA_TYPE_MESSAGE_ALARM,
                                                      DATA_TYPE_HAND_RISE_SWITCH,
                                                      DATA_TYPE_MESSAGE_ALARM,
                                                      DATA_TYPE_MESSAGE_SWITCH,

                                                      DATA_TYPE_CALL_ALARM,
                                                      DATA_TYPE_TARGET_ALARM,
                                                      DATA_TYPE_SITTING_REMIND,
                                                      DATA_TYPE_DRINK_ALARM,
                                                    ]
}


///: - 设置成功后，直接保存

extension YFHybridInfo {

    public func update(_ m: YFUnit)     { unitInfo = m; saveToCache() }
    public func update(_ m: YFHourly)   { hourlyInfo = m; saveToCache() }
    public func update(_ m: YFLanguage) { languageInfo = m; saveToCache() }

    public func update(_ m: YFUserInfo)  { userInfo = m; saveToCache() }
    public func update(_ m: YFClockInfo) { alarmInfo = m; saveToCache() }
    public func update(_ m: YFGoalInfo)         { goalInfo = m; saveToCache() }
    
    public func update(_ m: YFDisturb)              { disturbInfo = m; saveToCache() }
    public func update(_ m: YFHeartRateMonitor) { heartRateMonitor = m; saveToCache() }
    public func update(_ m: YFRaiseLight) { remindRaiseLight = m; saveToCache() }
    public func update(_ m: YFMessageNotify)        { messageNotifyInfo = m; saveToCache() }
    public func update(_ m: YFMessageNotifyType)   { remindNotifySwitch = m; saveToCache() }

    public func update(_ m: YFRemindCall)           { remindCall = m; saveToCache() }
    public func update(_ m: YFRemindTarget)         { remindTarget = m; saveToCache() }
    public func update(_ m: YFRemindSedentary)      { remindSedentary = m; saveToCache() }
    public func update(_ m: YFRemindDrink)          { remindDrink = m; saveToCache() }
}


