//
//  YFRecordType.swift
//  YFitKit
//
//  Created by Mac on 2021/4/23.
//

import Foundation
import HandyJSON
import YFitSDK

/// 总数据类型
public enum YFTotalType {
    case real       // 实时
    case history    // 历史
    
    public var name: String {
        switch self {
        case .real:    return "实时"
        case .history:    return "历史"
        }
    }
    
    static func parse(_ type: K6_DataFuncType) -> YFTotalType? {
        switch type {
        case .DATA_TYPE_REAL_SPORT: return .real
        case .DATA_TYPE_HISTORY_SPORT: return .history
        default: return nil
        }
    }
}

/// 心率数据类型
public enum YFHeartType {
    case real       // 实时
    case history    // 历史
    case exercise   // 训练

    public var name: String {
        switch self {
        case .real:    return "实时"
        case .history:    return "历史"
        case .exercise:    return "训练"
        }
    }
    
    static func parse(_ type: K6_DataFuncType) -> YFHeartType? {
        switch type {
        case .DATA_TYPE_REAL_HEART: return .real
        case .DATA_TYPE_HISTORY_HEART: return .history
        case .DATA_TYPE_EXERCISE_HEART: return .exercise
        default: return nil
        }
    }
}


/// 设备记录数据类型， 用于通知外部更新
public enum YFRecordType {
    case total(YFTotalType) // 运动数据
    case heart(YFHeartType) // 心率
    case sleep        // 睡眠
    case exercise     // 训练
    
    public var name: String {
        switch self {
        case .total(let type):    return "运动数据.\(type)"
        case .heart(let type):    return "心率数据.\(type)"
        case .sleep:    return "睡眠数据"
        case .exercise: return "训练数据"
        }
    }
}

extension K6_DataFuncType {
    var isRecord:Bool { K6_DataFuncType.recordCollection.contains(self) }
    static let recordCollection: [K6_DataFuncType] = [DATA_TYPE_REAL_SPORT, DATA_TYPE_HISTORY_SPORT,
                                                      DATA_TYPE_SLEEP,
                                                      DATA_TYPE_REAL_HEART, DATA_TYPE_HISTORY_HEART, DATA_TYPE_EXERCISE_HEART,
                                                      DATA_TYPE_MIX_SPORT]
}

extension YFRecordType {
    
    @discardableResult
    static func parse(_ type: K6_DataFuncType) -> YFRecordType? {
        switch type {
        case .DATA_TYPE_REAL_SPORT, .DATA_TYPE_HISTORY_SPORT:
            guard let totalType = YFTotalType.parse(type) else { return nil}
            return .total(totalType)
        case .DATA_TYPE_REAL_HEART, .DATA_TYPE_HISTORY_HEART, .DATA_TYPE_EXERCISE_HEART:
            guard let heartType = YFHeartType.parse(type) else { return nil}
            return .heart(heartType)
        case .DATA_TYPE_SLEEP: return .sleep
        case .DATA_TYPE_MIX_SPORT: return .exercise
        default: return nil
        }
    }
    
    /// 解析并保存，返回日期
    @discardableResult
    func parse(_ dict: Dictionary<String, Any>) -> String? {
        switch self {
        case .total(let type): return parse(total: dict, type: type)
        case .heart(let type): return parse(heart: dict, type: type)
        case .sleep: return parse(sleep: dict)
        case .exercise: return parse(exercise: dict)
        }
    }
    
    @discardableResult
    private func parse(total dict: Dictionary<String, Any>, type: YFTotalType) -> String? {
        guard let data = dict["Data"] as? Dictionary<String, Any> else { return nil }
        guard let array = data["sportInfos"] as? [Dictionary<String, Any>] else { return nil }
        guard var items = [YFTotal].deserialize(from: array)?.filter({ $0 != nil }) as? [YFTotal] else { return nil}
        items = items.sorted()
        items.forEach { $0.loadDateStr() }
        let savedArray = items.save()
        healthStoreSyncAction { YFHealthStore.save(sport: savedArray) }
        guard let first = savedArray.first else { return nil }
        return first.dateStr
    }
    
    @discardableResult
    private func parse(heart dict: Dictionary<String, Any>, type: YFHeartType) -> String? {
        guard let data = dict["Data"] as? Dictionary<String, Any> else { return nil }
        guard let array = data["heartInfos"] as? [Dictionary<String, Any>] else { return nil }
        guard var items = [YFHeart].deserialize(from: array)?.filter({ $0 != nil }) as? [YFHeart] else { return nil}
        items = items.sorted()
        items.forEach { $0.loadDateStr() }
        switch type {
        case .real, .history:
            let savedArray = items.save()
            YFHealthStore.save(heart: savedArray)
        case .exercise: items.saveExericise()
        }
        guard let first = items.first else { return nil}
        return first.dateStr
    }
    
    @discardableResult
    private func parse(sleep dict: Dictionary<String, Any>) -> String? {
        guard let data = dict["Data"] as? [Dictionary<String, Any>] else { return nil }
        guard let array1 = data.map({ $0["sleepInfos"] }) as? [[Dictionary<String, Any>]] else { return nil}
        var array2: [Dictionary<String, Any>] = []
        array1.forEach({ array2 += $0 })
        guard var items = [YFSleep].deserialize(from: array2)?.filter({ $0 != nil }) as? [YFSleep] else { return nil }
        items = items.filter({ $0.SleepStartTime > 0 }).sorted()
        
        items.forEach { $0.loadDateStr() }
        let savedArray = items.save()
        YFHealthStore.save(sleep: savedArray)
        guard let first = items.first else { return nil}
        return first.dateStr
    }
    
    private func parse(exercise dict: Dictionary<String, Any>) -> String? {
        guard let data = dict["Data"] as? [Dictionary<String, Any>] else { return nil }
        guard var items = [YFExerciseItem].deserialize(from: data)?.filter({ $0 != nil }) as? [YFExerciseItem] else { return nil }
        items = items.sorted()
        items.forEach { $0.loadDateStr() }
        items.save()
        guard let first = items.first else { return nil}
        return first.dateStr
    }
}
