//
//  YFHeart.swift
//  YFitKit
//
//  Created by Mac on 2021/1/31.
//

import Foundation
import HandyJSON

/// 一天的静态心率数据  ( 每10分钟记录一次，一天有144 次 )
public class YFHeart: YFModelBase {
    /// 日期
    public private(set) var dateStr: String = ""
    /// 时间戳
    public private(set) var time: TimeInterval = 0
    /// 心率
    public private(set) var heartNum: UInt8 = 0
    
    
    func loadDateStr() { dateStr = YFCalendarUtils.dateStr(timeInterval: time) }

}

extension YFHeart: Comparable {
    public static func < (lhs: YFHeart, rhs: YFHeart) -> Bool {
        lhs.time < rhs.time
    }
    
    public static func == (lhs: YFHeart, rhs: YFHeart) -> Bool {
        lhs.time == rhs.time
    }
}

///: - 数据库
/// 存储
extension Array where Element: YFHeart {
    // 静态心率 10 分钟
    @discardableResult
    func save() -> [YFHeart] {
        var array = [YFHeart]()
        forEach { if $0.save() { array.append($0)  } }
        return array
    }
    func saveExericise()  { forEach { $0.saveExericise()  } } // 训练心率 1 分钟

}

extension YFHeart {
    @discardableResult
    func save() -> Bool {  YFRecordHeart.save(with: self) }
    func saveExericise() {  YFRecordHeartExercise.save(with: self) }
}

/// 查询
extension YFHeart {
    
    static func parse(with record: YFRecordHeart) -> YFHeart {
        let model = YFHeart()
        model.dateStr = record.BG_dateStr
        model.time = record.BG_time
        model.heartNum = record.BG_heartNum
        return model
    }
    
    static func parse(with record: YFRecordHeartExercise) -> YFHeart {
        let model = YFHeart()
        model.dateStr = record.BG_dateStr
        model.time = record.BG_time
        model.heartNum = record.BG_heartNum
        return model
    }
    
    static func query(with date: Date) -> [YFHeart] {
        let records = YFRecordHeart.query(with: date)
        return records.map { YFHeart.parse(with: $0) }
    }
    
    static func query(start fromTime: TimeInterval) -> [YFHeart] {
        let records = YFRecordHeart.query(start: fromTime)
        return records.map { YFHeart.parse(with: $0) }
    }
    
    static func query(start fromTime: TimeInterval, end toTime: TimeInterval) -> [YFHeart] {
        let records = YFRecordHeart.query(start: fromTime, end: toTime)
        return records.map { YFHeart.parse(with: $0) }
    }
    
    static func queryExercise(start fromTime: TimeInterval, end toTime: TimeInterval) -> [YFHeart] {
        let records = YFRecordHeartExercise.query(start: fromTime, end: toTime)
        return records.map { YFHeart.parse(with: $0) }
    }
    
    public static func queryLatest(with date: Date) -> YFHeart? {
        guard let record = YFRecordHeart.queryLatest(with: date) else { return nil }
        return YFHeart.parse(with: record)
    }
}


//MARK: - 静态心率、睡眠心率

public enum YFHeartDayType {
    case detail
    case sleep
}

public class YFHeartDay {
    /// 类型
    public private(set) var type: YFHeartDayType = .detail
    /// 日期
    public private(set) var dateStr: String = ""
    /// 最新
    public private(set) var latest: UInt8 = 0
    /// 最高
    public private(set) var highest: UInt8 = 0
    /// 最低
    public private(set) var lowest: UInt8 = 0
    /// 最平均
    public private(set) var average: UInt8 = 0
    /// 每10分钟一个值， 没值填充 0
    public private(set) var values: [UInt8] = []
    
    ///: - Private
    
    private init(with type: YFHeartDayType, dateStr: String, for items: [YFHeart]) {
        self.type = type
        self.dateStr = dateStr
        self.items = items
        loadValues()
        loadDetail()
    }

    private var items: [YFHeart] = []
    
    private func loadValues() {
        
        guard items.count > 0 else { return }
        guard let dayDate = YFCalendarUtils.dateFormatter.date(from: dateStr)  else { return }
        let tenMinuteTime = TimeInterval(YFCalendarUtils.kSecondsTenMinutes)
        let oneHourTime = TimeInterval(YFCalendarUtils.kSecondsOneHour)

        /// 144 min values for static heart
        var count = 143
        var timeStart = dayDate.timeIntervalSince1970

        /// 96 min values for sleep heart
        if type == .sleep {
            count = 96
            timeStart = dayDate.timeIntervalSince1970 - (4 * oneHourTime)
        }
        
        var timeEnd = timeStart + tenMinuteTime
        for _ in 0...count {
            let items = items.filter { ($0.time >= timeStart && $0.time < timeEnd) }
            if let latestItem = items.last  {
                values.append(latestItem.heartNum)
            } else {
                values.append(0)
            }
            
            timeStart += tenMinuteTime
            timeEnd += tenMinuteTime
        }
    }
    
    private func loadDetail() {
        /// details
        let nonzeroValues = values.filter({ $0 != 0 })
        guard let last = nonzeroValues.last else { return }
        latest = last
        lowest = last
        highest = last
        var total: UInt32 = 0
        for value in nonzeroValues {
            total += UInt32(value)
            lowest = min(lowest, value)
            highest = max(highest, value)
        }
        average = UInt8(truncatingIfNeeded: total/UInt32(nonzeroValues.count))
    }
}

///: - 数据库
extension YFHeartDay {
    public static func query(details date: Date) -> YFHeartDay? {
        let items = YFHeart.query(with: date)
        guard items.count > 0 else { return nil }
        return YFHeartDay(with: .detail, dateStr: YFCalendarUtils.dateStr(date: date), for: items)
    }
    
    public static func query(sleep date: Date) -> YFHeartDay? {
        let oneHourTime = TimeInterval(YFCalendarUtils.kSecondsOneHour)
        let dayDate = YFCalendarUtils.dayDate(for: date)
        let start = dayDate.timeIntervalSince1970 - (4*oneHourTime)
        let end = dayDate.timeIntervalSince1970 + (12*oneHourTime)
        
        let items = YFHeart.query(start: start, end: end)
        guard items.count > 0 else { return nil }
        return YFHeartDay(with: .sleep, dateStr: YFCalendarUtils.dateStr(date: date), for: items)
    }
}

// MARK: - 运动心率

public class YFHeartExercise {
    /// 开始时间戳
    public private(set) var timeStart: TimeInterval = 0
    /// 结束时间戳
    public private(set) var timeEnd: TimeInterval = 0
    
    /// 最近
    public private(set) var latest: UInt8 = 0
    /// 最高
    public private(set) var highest: UInt8 = 0
    /// 最低
    public private(set) var lowest: UInt8 = 0
    /// 最平均
    public private(set) var average: UInt8 = 0
    /// 每1分钟一个值， 没值填充 0
    public private(set) var values: [UInt8] = []
    
    ///: - Private
    
    private init(form timeStart: TimeInterval, to timeEnd: TimeInterval, items: [YFHeart]) {
        self.timeStart = timeStart
        self.timeEnd = timeEnd
        self.items = items
        loadValues()
        loadDetail()
    }

    private var items: [YFHeart] = []
    
    private func loadValues() {
        
        guard items.count > 0 else { return }
        let oneMinuteTime = TimeInterval(YFCalendarUtils.kSecondsTenMinutes)
        var start = timeStart
        var end = timeStart + oneMinuteTime
        
        while end <= timeEnd {
            let items = items.filter { ($0.time >= timeStart && $0.time < timeEnd) }
            if let latestItem = items.last  {
                values.append(latestItem.heartNum)
            } else {
                values.append(0)
            }
            start += oneMinuteTime
            end += oneMinuteTime
        }
    }
    
    private func loadDetail() {
        /// details
        let nonzeroValues = values.filter({ $0 != 0 })
        guard let last = nonzeroValues.last else { return }
        latest = last
        lowest = last
        highest = last
        var total: UInt32 = 0
        for value in nonzeroValues {
            total += UInt32(value)
            lowest = min(lowest, value)
            highest = max(highest, value)
        }
        average = UInt8(truncatingIfNeeded: total/UInt32(nonzeroValues.count))
    }
}

///: - 数据库
extension YFHeartExercise {
    public static func query(frome start: TimeInterval, to end: TimeInterval) -> YFHeartExercise? {
        let items = YFHeart.queryExercise(start: start, end: end)
        guard items.count > 0 else { return nil }
        return YFHeartExercise(form: start, to: end, items: items)
    }
    
    public static func query(with exericise: YFExerciseItem) -> YFHeartExercise? {
        query(frome: exericise.start_time, to: exericise.end_time)
    }
}
