//
//  YFSleep.swift
//  YFitKit
//
//  Created by Mac on 2021/1/31.
//

import Foundation
import HandyJSON

public enum YFSleepType: UInt8, YFEnumtProtocol {
    case none = 0
    case start      // 开始睡眠
    case deep       // 深睡
    case light      // 浅睡
    case wakeup     // 睡醒
    
    var hkvalue: Int {
        switch self {
        case .none, .wakeup: return 0
        case .deep, .light: return 1
        case .start: return 2
        }
    }
}

/// 睡眠记录
public class YFSleep: YFModelBase {
    // 日期
    public private(set) var dateStr = ""
    // 时间戳
    public private(set) var SleepStartTime: TimeInterval = 0
    // 睡眠类型
    public private(set) var SleepType: YFSleepType = .none
    

    func loadDateStr() { dateStr = YFCalendarUtils.dateStr(timeInterval: SleepStartTime) }
}

extension YFSleep: Comparable {
    public static func < (lhs: YFSleep, rhs: YFSleep) -> Bool {
        lhs.SleepStartTime < rhs.SleepStartTime
    }
    
    public static func == (lhs: YFSleep, rhs: YFSleep) -> Bool {
        lhs.SleepStartTime == rhs.SleepStartTime
    }
}

///: - 数据库
/// 存储
extension Array where Element: YFSleep {
    @discardableResult
    func save() -> [YFSleep]  {
        var array = [YFSleep]()
        forEach { if $0.save() { array.append($0) }  }
        return array
    }
}

extension YFSleep {
    @discardableResult
    func save() -> Bool {  YFRecordSleep.save(with: self) }
}

/// 查询
extension YFSleep {
    static func parse(with record: YFRecordSleep) -> YFSleep {
        let model = YFSleep()
        model.dateStr = record.BG_dateStr
        model.SleepStartTime = record.BG_SleepStartTime
        model.SleepType = YFSleepType(rawValue: record.BG_SleepType) ?? .none
        return model
    }
    static func query(with date: Date) -> [YFSleep] {
        let records = YFRecordSleep.query(with: date)
        return records.map { YFSleep.parse(with: $0) }
    }
    
    static func query(start fromTime: TimeInterval) -> [YFSleep] {
        let records = YFRecordSleep.query(start: fromTime)
        return records.map { YFSleep.parse(with: $0) }
    }
    
    static func query(start fromTime: TimeInterval, end toTime: TimeInterval) -> [YFSleep] {
        let records = YFRecordSleep.query(start: fromTime, end: toTime)
        return records.map { YFSleep.parse(with: $0) }
    }
}


// MARK: -  睡眠详细 （20:00 ~ 12:00）
public class YFSleepDay {

    // 日期
    public private(set) var dateStr: String = ""
    /// Period of time
    public private(set) var periodTime: String = "--:--"
    /// 总时间长 （ 分钟 ）
    public private(set) var totalMinutes: UInt16 = 0
    /// 深睡时长（ 分钟 ）
    public private(set) var deeplMinutes: UInt16 = 0
    /// 浅睡时长（ 分钟 ）
    public private(set) var lightMinutes: UInt16 = 0
    /// 清醒时长（ 分钟 ）
    public private(set) var wakeMinutes: UInt16 = 0

    /// 一天的睡眠数据  ( 昨天 20:00 ~ 今天 12:00,  每分钟记录一次，总共 960 次 )
    public private(set) var records = [YFSleepType]()
    
    ///: - Private
    private init(with dateStr: String, for items: [YFSleep], details: Bool = false) {
        self.dateStr = dateStr
        self.items = items
        self.details = details
        loadDatas()
    }

    private var items: [YFSleep] = []
    private var details: Bool = false // 是否加载详情页 values 数据
    
    private func loadDatas() {
        guard items.count > 0 else { return }
        guard let dayDate = YFCalendarUtils.dateFormatter.date(from: dateStr)  else { return }
        let oneHourTime = TimeInterval(YFCalendarUtils.kSecondsOneHour)
        let PM_08_TIME = dayDate.timeIntervalSince1970 - (oneHourTime * 4);
        let AM_12_TIME = dayDate.timeIntervalSince1970 + (oneHourTime * 12);
        let records = items.filter({ $0.SleepStartTime >= PM_08_TIME && $0.SleepStartTime < AM_12_TIME })
        guard let first = records.first, let last = records.last else { return }
        self.periodTime = "\(YFCalendarUtils.timeStr(timeInterval: first.SleepStartTime))-\(YFCalendarUtils.timeStr(timeInterval: last.SleepStartTime))"
        loadValues(type: .none, count: YFCalendarUtils.minutesOffset(from: PM_08_TIME, to: first.SleepStartTime))
        
        var previous: YFSleep? = nil
        for record in records {
            if let hasPrevious = previous {
                loadTotal(type: hasPrevious.SleepType, count: YFCalendarUtils.minutesOffset(from: hasPrevious.SleepStartTime, to: record.SleepStartTime))
            }
            previous = record
        }
        
        loadValues(type: .none, count: YFCalendarUtils.minutesOffset(from: last.SleepStartTime, to: AM_12_TIME))

    }
    
    private func loadTotal(type: YFSleepType, count: Int) {
        if type == .start { self.wakeMinutes += UInt16(count) }
        if type == .deep  { self.deeplMinutes += UInt16(count) }
        if type == .light { self.lightMinutes += UInt16(count) }
        if type != .wakeup { self.totalMinutes += UInt16(count) } 
        loadValues(type: type, count: count)
    }
    
    private func loadValues(type: YFSleepType, count: Int) {
        guard details else { return }
        records += [YFSleepType](repeating: type, count: count)
    }
}

///: - 数据库
extension YFSleepDay {
    public static func query(with date: Date, details: Bool = false) -> YFSleepDay? {
        let dayDate = YFCalendarUtils.dayDate(for: date)
        let oneHourTime = TimeInterval(YFCalendarUtils.kSecondsOneHour)
        let PM_08_TIME = dayDate.timeIntervalSince1970 - (oneHourTime * 4);
        let AM_12_TIME = dayDate.timeIntervalSince1970 + (oneHourTime * 12);
        let records = YFSleep.query(start: PM_08_TIME, end: AM_12_TIME)
        return YFSleepDay(with: YFCalendarUtils.dateStr(date: date), for: records, details: details)
    }
    
    public static func query(from fromDate: Date, to toDate: Date) -> [YFSleepDay] {
        var resultArray = [YFSleepDay]()
        var date = fromDate
        while date <= toDate {
            if let item = query(with: date) {
                resultArray.append(item)
            } else {
                // resultArray.append(YFTotal())
            }
            if let nextDate = YFCalendarUtils.offsetDay(date, 1) {
                date = nextDate
            } else {
                break;
            }
        }
        return resultArray
    }
}


// MARK: -  持续睡眠
class YFSleepDuration {
    // 日期
    public private(set) var dateStr = ""
    // 时间戳
    public private(set) var startTime: TimeInterval = 0
    // 时间戳
    public private(set) var endTime: TimeInterval = 0
    // 睡眠类型
    public private(set) var type: YFSleepType = .none
    // 持续睡眠时间（分钟）
    public private(set) var minutes: Int = 0
    
    ///: - Private
    private init(for start: YFSleep, to end: YFSleep) {
        self.dateStr = start.dateStr
        self.startTime = start.SleepStartTime
        self.endTime = end.SleepStartTime
        self.type = start.SleepType
        self.minutes = YFCalendarUtils.minutesOffset(from: start.SleepStartTime, to: end.SleepStartTime)
    }
}


///: - 数据库
extension YFSleepDuration {
    
    static func parse(with records: [YFSleep]) -> [YFSleepDuration] {
        var previous: YFSleep? = nil
        var array = [YFSleepDuration]()
        for record in records {
            if let hasPrevious = previous {
                let duration = YFSleepDuration(for: hasPrevious, to: record)
                if duration.minutes > 0, duration.type != .none {
                    array.append(duration)
                }
            }
            previous = record
        }
        return array
    }
    
    static func query(today date: Date) -> [YFSleepDuration] {
        let dayDate = YFCalendarUtils.dayDate(for: date)
        let oneHourTime = TimeInterval(YFCalendarUtils.kSecondsOneHour)
        let PM_08_TIME = dayDate.timeIntervalSince1970 - (oneHourTime * 4);
        let AM_12_TIME = dayDate.timeIntervalSince1970 + (oneHourTime * 12);
        let records = YFSleep.query(start: PM_08_TIME, end: AM_12_TIME)
        return parse(with: records)
    }
}
