//
//  YFTotal.swift
//  YFitKit
//
//  Created by Mac on 2021/4/12.
//

import Foundation
import HandyJSON

// MARK: - 手表运动总数据， 运动某时间段后的当天运动总步数、总距离、总卡路里消耗
public class YFTotal: YFModelBase {
    /// 日期
    public private(set) var dateStr: String = ""
    /// 开始时间戳
    public private(set) var startSecs: TimeInterval = 0
    /// 持续用时
    public private(set) var walkDuration: UInt32 = 0
    
    /// 手表给到的是当天总值，而不是区间值
    /// 步数
    public private(set) var walkSteps: UInt32 = 0
    /// 距离（m）
    public private(set) var walkDistance: UInt32 = 0
    /// 卡路里 (kCal)
    public private(set) var walkCalories: UInt32 = 0
    
    func loadDateStr() { dateStr = YFCalendarUtils.dateStr(timeInterval: startSecs) }
}

extension YFTotal: Comparable {
    public static func < (lhs: YFTotal, rhs: YFTotal) -> Bool {
        lhs.startSecs < rhs.startSecs
    }
    
    public static func == (lhs: YFTotal, rhs: YFTotal) -> Bool {
        lhs.startSecs == rhs.startSecs
    }
}

///: - 数据库
/// 存储
extension Array where Element: YFTotal {
    @discardableResult
    func save() -> [YFTotal] {
        var array = [YFTotal]()
        forEach { if $0.save() { array.append($0) } }
        return array
    }
}

extension YFTotal {
    @discardableResult
    func save() -> Bool {  YFRecordTotal.save(with: self) }
}

/// 查询
extension YFTotal {
    static func parse(with record: YFRecordTotal) -> YFTotal {
        let model = YFTotal()
        model.dateStr = record.BG_dateStr
        model.startSecs = record.BG_startSecs
        model.walkDuration = record.BG_walkDuration
        model.walkSteps = record.BG_walkSteps
        model.walkDistance = record.BG_walkDistance
        model.walkCalories = record.BG_walkCalories
        return model
    }
    static func query(with date: Date) -> [YFTotal] {
        let records = YFRecordTotal.query(with: date)
        return records.map { YFTotal.parse(with: $0) }
    }
    static func query(start fromTime: TimeInterval) -> [YFTotal] {
        let records = YFRecordTotal.query(start: fromTime)
        return records.map { YFTotal.parse(with: $0) }
    }
    
    /// 查询某时间戳之前最近的一条记录
    static func queryLatest(before time: TimeInterval) -> YFTotal? {
        guard let record = YFRecordTotal.queryLatest(before: time) else { return nil }
        return YFTotal.parse(with: record)
    }
    
    /// 查询某天总数据
    public static func queryLatest(with date: Date) -> YFTotal? {
        guard let record = YFRecordTotal.queryLatest(with: date) else { return nil }
        return YFTotal.parse(with: record)
    }
    
    /// 查询【fromDate，toDate 】天总数据
    public static func query(from fromDate: Date, to toDate: Date) -> [YFTotal] {
        var resultArray = [YFTotal]()
        var date = fromDate
        while date <= toDate {
            if let item = queryLatest(with: date) {
                resultArray.append(item)
            } else {
                // resultArray.append(YFTotal())
            }
            if let nextDate = YFCalendarUtils.offsetDay(date, 1) {
                date = nextDate
            } else {
                break;
            }
        }
        return resultArray
    }
}


// MARK: - 某天的显示数据

public class YFTotalDay {
    /// 日期
    public private(set) var dateStr: String = ""
    /// 最后一项，表示一天的总数据
    public var latest: YFTotal? { allItems.last }
    /// 小时区间项 ，表示24小时时间区间步数
    public private(set) var hourIntervalItems: [YFSportInterval] = []

    private init(with dateStr: String, for items: [YFTotal]) {
        self.dateStr = dateStr
        self.allItems = items
        loadHourSteps()
    }
    
    private(set) var allItems: [YFTotal] = []

    private func spacingValue(_ v1: UInt32, _ v2: UInt32) -> UInt32{
        (v2 >= v1 ? (v2 - v1) : v2)
    }
    
    /// 解析出24小时区间值
    private func loadHourSteps() {
        guard allItems.count > 0 else { return }
        guard let date = YFCalendarUtils.dateFormatter.date(from: dateStr)  else { return }
        /// 小时时间段
        let hourTime = TimeInterval(YFCalendarUtils.kSecondsOneHour)
        var timeStart = date.timeIntervalSince1970
        var timeEnd = timeStart + hourTime
        /// 上一小时项
        var lastHourItem: YFTotal? = nil
        for _ in 0...23 {
            let hourIntervalItem = YFSportInterval()
            hourIntervalItem.dateStr = dateStr
            hourIntervalItem.startSecs = timeStart
            hourIntervalItem.walkDuration = UInt32(YFCalendarUtils.kSecondsOneHour)
            let hourItems = allItems.filter { ($0.startSecs >= timeStart && $0.startSecs < timeEnd) }
            if let hourLatestItem = hourItems.last  {
                if let hasLastHourItem = lastHourItem {
                    hourIntervalItem.walkSteps = spacingValue(hasLastHourItem.walkSteps, hourLatestItem.walkSteps)
                    hourIntervalItem.walkDistance = spacingValue(hasLastHourItem.walkDistance, hourLatestItem.walkDistance)
                    hourIntervalItem.walkCalories = spacingValue(hasLastHourItem.walkCalories, hourLatestItem.walkCalories)
                } else {
                    hourIntervalItem.walkSteps = hourLatestItem.walkSteps
                    hourIntervalItem.walkDistance = hourLatestItem.walkDistance
                    hourIntervalItem.walkCalories = hourLatestItem.walkCalories
                }
                lastHourItem = hourLatestItem
            }
            hourIntervalItems.append(hourIntervalItem)
            timeStart += hourTime
            timeEnd += hourTime
        }
    }
}


///: - 数据库
extension YFTotalDay {
    /// 数据库查询某天运动数据详情
    public static func query(with date: Date) -> YFTotalDay? {
        let items = YFTotal.query(with: date)
        guard items.count > 0 else { return nil}
        return YFTotalDay(with: YFCalendarUtils.dateStr(date: date), for: items)
    }
}



// MARK: - 运动区间， 某时间段的运动步数、距离、卡路里消耗
public class YFSportInterval: YFModelBase {
    /// 日期
    public var dateStr: String = ""
    /// 开始时间戳
    public var startSecs: TimeInterval = 0
    /// 持续用时
    public var walkDuration: UInt32 = 0
    
    /// 步数
    public var walkSteps: UInt32 = 0
    /// 距离（m）
    public var walkDistance: UInt32 = 0
    /// 卡路里 (kCal)
    public  var walkCalories: UInt32 = 0

}

///: - 数据库
extension YFSportInterval {
    ///: - 某天的第一组值
    /// 可能出现异常数据（前面有记录丢失）。如已推送到别的手机
    /// startSecs 为零点时间，walkDuration计算为
    private static func parse(with m: YFTotal) -> YFSportInterval? {
        let interval = YFSportInterval()
        interval.dateStr = m.dateStr
        interval.startSecs =  m.startSecs
        interval.walkDuration = m.walkSteps
        interval.walkSteps = m.walkSteps
        interval.walkDistance = m.walkDistance
        interval.walkCalories = m.walkCalories
        guard interval.walkSteps > 0 else { return nil}
        return interval
    }
    
    private static func spacingValue(_ v1: UInt32, _ v2: UInt32) -> UInt32{
        (v2 >= v1 ? (v2 - v1) : v2)
    }
    
    private static func parse(with m1: YFTotal, and m2: YFTotal) -> YFSportInterval? {
        let interval = YFSportInterval()
        interval.dateStr = m2.dateStr
        interval.startSecs = m2.startSecs
        interval.walkDuration =  spacingValue(m1.walkDuration, m2.walkDuration)
        interval.walkSteps = spacingValue(m1.walkSteps, m2.walkSteps)
        interval.walkDistance = spacingValue(m1.walkDistance, m2.walkDistance)
        interval.walkCalories = spacingValue(m1.walkCalories, m2.walkCalories)
        guard interval.walkSteps > 0 else { return nil}
        return interval
    }

    static func parse(with items: [YFTotal]) -> [YFSportInterval] {
        var resultArray: [YFSportInterval] = []
        guard var lastItem = items.first else { return resultArray }
        for (idx, item) in items.enumerated() {
            if idx == 0 {
                if let m = YFSportInterval.parse(with: lastItem) {
                    if m.walkDuration > 60 {  m.walkDuration = 0 }
                    resultArray.append(m)
                }
            } else {
                let sameDay = YFCalendarUtils.someDay(from: lastItem.startSecs, to: item.startSecs)
                if sameDay {
                    if let m = YFSportInterval.parse(with: lastItem, and: item) { resultArray.append(m) }
                } else {
                    if let m = YFSportInterval.parse(with: item) { resultArray.append(m) }
                }
            }
            lastItem = item
        }
        return resultArray
    }
}
