//
//  YFCommand.swift
//  YFitKit
//
//  Created by Mac on 2021/1/31.
//

import Foundation
import YFitSDK

public typealias YFWriteClosure = ((Error?) -> Void)

public class YFCommand {
    
    fileprivate init(_ cmd: CE_Cmd = CE_Cmd()) {
        self.cmd = cmd
    }
    
    public func writeToDevice(_ callback: @escaping YFWriteClosure) { _writeToDevice(callback) }
    
    public func writeToDevice() { _writeToDevice(nil) }
        
    /// - Private
    
    private var cmd: CE_Cmd
    
    private var nameText: String?
    
    private var descripteText: String?

    private func loggerPrint(_ error: Error?) {
        // if YFReponseError.invaild(error) { return  }
        var textLogger = "Sended: \(nameText ?? "未知指令")"
        if let text = descripteText { textLogger += "\nParsed: \(text)"}
        guard let err = error else { YFLog.v(textLogger); return }
        textLogger += "\nError: \(err)"
        YFLog.e(textLogger)
    }
    
    
    private func _writeToDevice(_ callback: YFWriteClosure? = nil) {
        YFCommand._writeToDevice(self, callback)
    }
    
    private static func _writeToDevice(_ cmd: YFCommand, _ callback: YFWriteClosure? = nil) {
        YFKitService.sendCmdToDevice(cmd: cmd.cmd) { (error) in
            cmd.loggerPrint(error)
            callback?(error)
        }
    }
}

// MARK: - 设置指令

extension YFCommand {
    
    static var hybridInfo: YFHybridInfo { YFDataManager.hybridInfo }
    
    ///: -
    /// 设置单位
    public static func setUnit(_ unit: YFUnit) -> YFCommand {
        let command = YFCommand(unit.cmd)
        command.nameText = "保存单位设置"
        command.descripteText = "\(unit)"
        return command
    }
    /// 设置时间制
    public static func setHourly(_ hourly: YFHourly) -> YFCommand {
        let cmd = hourly.cmd
        let command = YFCommand(cmd)
        command.nameText = "设置时间制"
        command.descripteText = "\(hourly)"
        return command
    }
    /// 设置语言
    public static func setLanguage(_ language: YFLanguage) -> YFCommand {
        let cmd = language.cmd
        let command = YFCommand(cmd)
        command.nameText = "设置语言"
        command.descripteText = "\(language)"
        return command
    }
    
    /// : -
    /// 用户信息
    public static func setUserInfo(_ user: YFUserInfo) -> YFCommand {
        let command = YFCommand(user.cmd)
        command.nameText = "保存用户设置"
        command.descripteText = "\(user)"
        return command
    }
    /// 闹钟
    public static func setClockInfo(_ clock: YFClockInfo) -> YFCommand {
        let command = YFCommand(clock.cmd)
        command.nameText = "保存闹钟设置"
        command.descripteText = "\(clock)"
        return command
    }
    /// 设置目标提醒值
    public static func setGoalInfo(_ goalInfo: YFGoalInfo) -> YFCommand {
        let command = YFCommand(goalInfo.cmd)
        command.nameText = "设置目标提醒值"
        command.descripteText = "\(goalInfo)"
        return command
    }
    
    /// : -
    /// 设置免打扰时间段
    public static func setDisturb(_ distrub: YFDisturb) -> YFCommand {
        let command = YFCommand(distrub.cmd)
        command.nameText = "设置免打扰时间段"
        command.descripteText = "\(distrub)"
        return command
    }
    /// 设置静态心率检测，只有开关
    public static func setHeartRateMonitor(_ heartRateMonitor: YFHeartRateMonitor) -> YFCommand {
        let command = YFCommand(heartRateMonitor.cmd)
        command.nameText = "设置静态心率检测.\(heartRateMonitor.onoff.text)"
        return command
    }
    /// 设置翻腕亮屏
    public static func setRaiseLight(_ raiseLight: YFRaiseLight) -> YFCommand {
        let command = YFCommand(raiseLight.cmd)
        command.nameText = "设置翻腕亮屏"
        command.descripteText = "\(raiseLight)"
        return command
    }
    /// 设置消息通知, 只有开关
    public static func setMessageNotify(_ messageNotify: YFMessageNotify) -> YFCommand {
        let command = YFCommand(messageNotify.cmd)
        command.nameText = "设置消息推送.\(messageNotify.onoff.text)"
        return command
    }
    /// 设置消息通知，应用开关
    public static func setMessageNotifyType(_ notifySwitch: YFMessageNotifyType) -> YFCommand {
        let command = YFCommand(notifySwitch.cmd)
        command.nameText = "设置应用推送开关"
        command.descripteText = "\(notifySwitch)"
        return command
    }
    
    /// : -
    /// 设置来电提醒，只有开关
    public static func setRemindCall(_ remindCall: YFRemindCall) -> YFCommand {
        let command = YFCommand(remindCall.cmd)
        command.nameText = "设置来电提醒.\(remindCall.onoff.text)"
        return command
    }
    
    /// 设置目标提醒，只有开关
    public static func setRemindTarget(_ remindTarget: YFRemindTarget) -> YFCommand {
        let command = YFCommand(remindTarget.cmd)
        command.nameText = "设置目标提醒.\(remindTarget.onoff.text)"
        return command
    }
    /// 设置久坐提醒
    public static func setRemindSedentary(_ sedentary: YFRemindSedentary) -> YFCommand {
        let command = YFCommand(sedentary.cmd)
        command.nameText = "设置久坐提醒"
        command.descripteText = "\(sedentary)"
        return command
    }
    /// 设置喝水提醒
    public static func setRemindDrink(_ drink: YFRemindDrink) -> YFCommand {
        let command = YFCommand(drink.cmd)
        command.nameText = "设置喝水提醒"
        command.descripteText = "\(drink)"
        return command
    }
    
    /// : -
    /// 混杂设置、以及返回
    public static func syncHybridInfo(_ cmd: CE_SyncHybridCmd) -> YFCommand {
        let command = YFCommand()
        command.cmd = cmd
        return command
    }
}

// MARK: - 控制指令、同步指令

extension YFCommand {
    /// 配对指令
    public static func pair() -> YFCommand {
        let cmd = CE_SyncPairOKCmd()
        cmd.firstPairStatus = 1
        let command = YFCommand(cmd)
        command.nameText = "配对指令"
        return command
    }
    
    /// App 退出指令
    public static func appExit() -> YFCommand {
        let command = YFCommand(CE_AppExitCmd())
        command.nameText = "App退出"
        return command
    }
    
    /// 恢复出厂
    public static func reset() -> YFCommand {
        let command = YFCommand(CE_SyncResetCmd())
        command.nameText = "恢复出厂"
        return command
    }
    
    /// 数据传输开关，关闭时不会有数据接收
    public static func appSensor(_ type: YFCmdSwitchType) -> YFCommand {
        let cmd = CE_SensorCmd()
        cmd.onoff = type.rawValue
        let command = YFCommand(cmd)
        command.nameText = "数据传输传感器.\(type.text)"
        return command
    }
    
    /// 查找手环
    public static func findBand(_ type: YFCmdSwitchType) -> YFCommand {
        let cmd = YD_SyncFindDevCmd()
        cmd.onoff = type.rawValue
        let command = YFCommand(cmd)
        command.nameText = "查找手环.\(type.text)"
        return command
    }
    
    /// 相机
    public static func camera(_ type: YFCmdSwitchType) -> YFCommand {
        let cmd = CE_SendPhotoCmd()
        cmd.onoff = type.rawValue
        let command = YFCommand(cmd)
        command.nameText = "手表相机.\(type.text)"
        return command
    }
    
    /// 同步时间
    public static func syncTime() -> YFCommand {
        let cmd = hybridInfo.hourlyInfo.cmd
        let command = YFCommand(cmd)
        command.nameText = "同步时间"
        return command
    }
    
    /// 同步语言
    public static func syncLanguage() -> YFCommand {
        let language = YFLanguage()
        language.type = YFLanType.app
        let command = YFCommand(language.cmd)
        command.nameText = "同步语言.\(language)"
        return command
    }
    
    /// 同步agps
    public static func agps(_ sport: YFSportAgps) -> YFCommand {
        let command = YFCommand(sport.cmd)
        command.nameText = "同步Agps"
        command.descripteText = "\(sport)"
        return command
    }
    
    /// 同步天气
    public static func weather(_ index: Int) -> YFCommand? {
        guard let weather = YFWeather.weather(index) else { return nil }
        let command = YFCommand(weather.cmd)
        command.nameText = "同步天气:\(index)"
        command.descripteText = "\(weather)"
        return command
    }
}


// MARK: - 读取指令

extension YFCommand {
    /// 读取配对转态
    public static func readPairInfo() -> YFCommand {
        let command = YFCommand(CE_RequestSystemPairStatusCmd())
        command.nameText = "读取配对信息"
        return command
    }

    /// 目标
    public static func readGoalInfo() -> YFCommand {
        let command = YFCommand(CE_RequestGoalCmd())
        command.nameText = "读取目标"
        return command
    }
    
    /// OTA 设备
    public static func readOTADeviceInfo() -> YFCommand {
        let command = YFCommand(CE_RequestDevInfoCmd())
        command.nameText = "读取OTA设备"
        return command
    }
    
    /// OTA 状态
    public static func readOTAStatus() -> YFCommand {
        let command = YFCommand(CE_RequestOTAStatusCmd())
        command.nameText = "读取OTA状态"
        return command
    }
    
    /// 用户信息
    public static func readUserInfo() -> YFCommand {
        let command = YFCommand(CE_RequestUserInfo())
        command.nameText = "读取用户信息"
        return command
    }
    
    /// 读取混杂数据
    public static func readHybrid() -> YFCommand {
        let command = YFCommand(CE_SyncHybridCmd())
        command.nameText = "读取混杂数据"
        return command
    }
}



