//
//  YFRecordExercise.swift
//  YFitKit
//
//  Created by Mac on 2021/5/12.
//

import Foundation
import GRDB

struct YFRecordExercise: Codable {
    var BG_bg_id: Int?
    var BG_bg_createTime: String = ""
    var BG_bg_updateTime: String = ""

    var BG_dateStr: String = ""
    var BG_sport_type: UInt8 = 0
    var BG_start_time: TimeInterval = 0
    var BG_end_time: TimeInterval = 0
    var BG_total_time: UInt16 = 0
    
    var BG_step: UInt32 = 0
    var BG_distance: UInt16 = 0
    var BG_calories: UInt16 = 0
    var BG_speed: Double = 0
    var BG_max_speed: Double = 0
    var BG_heartrate: UInt8 = 0
    var BG_max_hr: UInt8 = 0
    var BG_aver_hr: UInt8 = 0
        
    init(with model: YFExerciseItem, isUpdate:Bool = false) {
        self.BG_dateStr = model.dateStr
        self.BG_sport_type = model.sport_type.rawValue
        self.BG_start_time = model.start_time
        self.BG_end_time = model.end_time
        self.BG_total_time = model.total_time
        
        self.BG_step = model.step
        self.BG_distance = model.distance
        self.BG_calories = model.calories
        self.BG_speed = model.speed
        self.BG_max_speed = model.max_speed
        self.BG_heartrate = model.heartrate
        self.BG_max_hr = model.max_hr
        self.BG_aver_hr = model.aver_hr

        let nowStr = YFCalendarUtils.datetimeStr(date: Date())
        if !isUpdate { self.BG_bg_createTime = nowStr }
        self.BG_bg_updateTime = nowStr
    }
}

// MARK: - ColumnExpression
extension YFRecordExercise {
    enum Columns: String, ColumnExpression {
        case BG_bg_id, BG_bg_createTime, BG_bg_updateTime
        case BG_dateStr, BG_sport_type
        case BG_start_time, BG_end_time, BG_total_time
        
        case BG_step, BG_distance, BG_calories
        case BG_speed, BG_max_speed
        case BG_heartrate, BG_max_hr, BG_aver_hr
    }
}

// MARK: - TableRecord
/// 使用 TableRecord 协议, 告知操作数据表
extension YFRecordExercise: TableRecord {
    //  数据表名字 如果不自定义 将使用默认的表名, 默认为小写开头驼峰命名 例如 UserEntity -> userEntity
    public static var databaseTableName: String { return "BHYSportExercise" }
    // 声明列名表示的方式
    public static var databaseSelection: [SQLSelectable] = [Columns.BG_bg_id, Columns.BG_bg_createTime, Columns.BG_bg_updateTime,
                                                            Columns.BG_dateStr, Columns.BG_sport_type,
                                                            Columns.BG_start_time, Columns.BG_end_time, Columns.BG_total_time,
                                                            Columns.BG_step, Columns.BG_distance, Columns.BG_calories,
                                                            Columns.BG_speed, Columns.BG_max_speed,
                                                            Columns.BG_heartrate, Columns.BG_max_hr, Columns.BG_aver_hr
                                                            ]
}

// MARK: - FetchableRecord
/// FetchableRecord 进行查询操作
/// 使用了Codable可以不实现  init (row: Row)
/// 未使用 Codable = Decodable & Encodable 协议的需要实现 init (row: Row)
extension YFRecordExercise: FetchableRecord {
    public init(row: Row) {
        BG_bg_id = row[Columns.BG_bg_id]
        BG_bg_createTime = row[Columns.BG_bg_createTime]
        BG_bg_updateTime = row[Columns.BG_bg_updateTime]
        
        BG_dateStr = row[Columns.BG_dateStr]
        BG_sport_type = row[Columns.BG_sport_type]
        BG_start_time = row[Columns.BG_start_time]
        BG_end_time = row[Columns.BG_end_time]
        BG_total_time = row[Columns.BG_total_time]
        
        BG_step = row[Columns.BG_step]
        BG_distance = row[Columns.BG_distance]
        BG_calories = row[Columns.BG_calories]
        BG_speed = row[Columns.BG_speed]
        BG_max_speed = row[Columns.BG_max_speed]
        BG_heartrate = row[Columns.BG_heartrate]
        BG_max_hr = row[Columns.BG_max_hr]
        BG_aver_hr = row[Columns.BG_aver_hr]
    }
}

// MARK: - MutablePersistableRecord
/// 使用PersistableRecord / MutablePersistableRecord插入更新保存数据,
/// 存储模型是Class使用PersistableRecord,
/// 存储模型是struct使用MutablePersistableRecord.
/// 两者区别在于 MutablePersistableRecord save() insert() 是 mutating.
extension YFRecordExercise: MutablePersistableRecord {
    
    public func encode(to container: inout PersistenceContainer) {
        container[Columns.BG_bg_id] = BG_bg_id
        container[Columns.BG_bg_createTime] = BG_bg_createTime
        container[Columns.BG_bg_updateTime] = BG_bg_updateTime

        container[Columns.BG_dateStr] = BG_dateStr
        container[Columns.BG_sport_type] = BG_sport_type
        container[Columns.BG_start_time] = BG_start_time
        container[Columns.BG_end_time] = BG_end_time
        container[Columns.BG_total_time] = BG_total_time

        container[Columns.BG_step] = BG_step
        container[Columns.BG_distance] = BG_distance
        container[Columns.BG_calories] = BG_calories
        container[Columns.BG_speed] = BG_speed
        container[Columns.BG_max_speed] = BG_max_speed
        container[Columns.BG_heartrate] = BG_heartrate
        container[Columns.BG_max_hr] = BG_max_hr
        container[Columns.BG_aver_hr] = BG_aver_hr
    }
    
    mutating public func didInsert(with rowID: Int64, for column: String?) {
        
    }
    
}

extension YFRecordExercise {
    
    /// 获取数据库对象
    private static var dbQueue: DatabaseQueue? {
        SQLiteManager.dbQueue
    }
    
    /// 创建数据库
    private static func createTable() -> Void {
        guard let _dbQueue = dbQueue else { return }
        try! _dbQueue.inDatabase { (db) -> Void in
            /// 判断是否存在数据库
            if try db.tableExists(databaseTableName) { return }
            
            /// 创建数据库表
            try db.create(table: databaseTableName, temporary: false, ifNotExists: true, body: { (t) in
                t.autoIncrementedPrimaryKey(Columns.BG_bg_id.rawValue)
                t.column(Columns.BG_bg_createTime.rawValue, .text)
                t.column(Columns.BG_bg_updateTime.rawValue, .text)
                
                t.column(Columns.BG_dateStr.rawValue,.integer)
                t.column(Columns.BG_sport_type.rawValue,.integer)
                t.column(Columns.BG_start_time.rawValue,.double)
                t.column(Columns.BG_end_time.rawValue,.double)
                t.column(Columns.BG_total_time.rawValue,.integer)

                t.column(Columns.BG_step.rawValue,.integer)
                t.column(Columns.BG_distance.rawValue,.integer)
                t.column(Columns.BG_calories.rawValue,.integer)
                t.column(Columns.BG_speed.rawValue,.integer)
                t.column(Columns.BG_max_speed.rawValue,.double)
                t.column(Columns.BG_heartrate.rawValue,.double)
                t.column(Columns.BG_max_hr.rawValue,.integer)
                t.column(Columns.BG_aver_hr.rawValue,.integer)

                t.uniqueKey([Columns.BG_start_time.rawValue])
            })
        }
    }
}

// MARK: - 插入&更新

extension YFRecordExercise {
    
    private static var now: Date { Date() }
    
    private static var _latestRecord: YFRecordExercise?
    
    /// 查询
    private static func queryRecordLatest() -> YFRecordExercise? {
        let records = queryAllRecord(limit: 1)
        return records.first
    }
    
    /// 更新
    private static func update(_ record: YFRecordExercise, _ item: YFExerciseItem) {
        guard let _dbQueue = dbQueue else { return }
        /// 创建数据库表
        self.createTable()
        /// 事务 更新场景
        try! _dbQueue.inTransaction { (db) -> Database.TransactionCompletion in
            do {
                var temp = record
                temp.BG_sport_type = item.sport_type.rawValue
                temp.BG_start_time = item.start_time
                temp.BG_end_time = item.end_time
                temp.BG_total_time = item.total_time
                temp.BG_step = item.step
                temp.BG_distance = item.distance
                temp.BG_calories = item.calories
                temp.BG_speed = item.speed
                temp.BG_max_speed = item.max_speed
                temp.BG_heartrate = item.heartrate
                temp.BG_max_hr = item.max_hr
                temp.BG_aver_hr = item.aver_hr
                temp.BG_bg_updateTime = YFCalendarUtils.datetimeStr(date: now)
                try temp.update(db)
                return Database.TransactionCompletion.commit
            } catch {
                return Database.TransactionCompletion.rollback
            }
        }
    }
    
    /// 插入
    private static func insert(_ item: YFExerciseItem) {
        guard let _dbQueue = dbQueue else { return }
        /// 创建数据库表
        self.createTable()
        let record = YFRecordExercise(with: item, isUpdate: false)
        /// 事务
        try! _dbQueue.inTransaction { (db) -> Database.TransactionCompletion in
            do {
                var temp = record
                try temp.insert(db)
                return Database.TransactionCompletion.commit
            } catch {
                return Database.TransactionCompletion.rollback
            }
        }
    }
    
    /// 保存单个数据， dateStr 相等则更新， 否则插入
    @discardableResult
    internal static func save(with item: YFExerciseItem) -> Bool {
        guard item.start_time > YFCalendarUtils.date(from: "2010", format: "yyyy").timeIntervalSince1970 else { return false}
        guard item.start_time < now.timeIntervalSince1970 + 10 else { return false}
        if let record =  latestRecord,  item.start_time < record.BG_start_time { return false}
        _latestRecord = YFRecordExercise(with: item)

        /// 判断是否存在
        if let record = queryRecord(time: item.start_time) {
            self.update(record, item)
        } else {
            self.insert(item)
        }
        return true
    }
}



// MARK: - 删除
extension YFRecordExercise {
    
    internal static func remove() {
        guard let _dbQueue = dbQueue else { return }
        /// 事务
        try! _dbQueue.inTransaction { (db) -> Database.TransactionCompletion in
            do {
                try db.drop(table: databaseTableName)
                return Database.TransactionCompletion.commit
            } catch {
                return Database.TransactionCompletion.rollback
            }
        }
    }
    
    /// 删除所有
    internal static func deleteAll() {
        _latestRecord = nil
        guard let _dbQueue = dbQueue else { return }
        /// 是否有数据库表
        self.createTable()
        /// 事务
        try! _dbQueue.inTransaction { (db) -> Database.TransactionCompletion in
            do {
                try YFRecordExercise.deleteAll(db)
                return Database.TransactionCompletion.commit
            } catch {
                return Database.TransactionCompletion.rollback
            }
        }
    }
}


// MARK: - 查询

extension YFRecordExercise {
    
    private static func dateStr(_ date: Date) -> String {
        return YFCalendarUtils.dateStr(date: date)
    }
    
    /// 查询所有: 最近limit条记录
    private static func queryAllRecord(limit: Int = Int.max) -> [YFRecordExercise] {
        guard let _dbQueue = dbQueue else { return [] }
        /// 创建数据库
        self.createTable()
        /// 返回查询结果
        return try! _dbQueue.unsafeRead({ (db) -> [YFRecordExercise] in
            return try YFRecordExercise.order(Columns.BG_start_time).reversed().limit(limit).fetchAll(db)
        })
    }
    
    /// 查询指定时间戳
    private static func queryRecord(time: TimeInterval) -> YFRecordExercise? {
        guard let _dbQueue = dbQueue else { return nil }
        /// 创建数据库
        self.createTable()
        /// 返回查询结果
        return try! _dbQueue.unsafeRead { (db) -> YFRecordExercise? in
            return try YFRecordExercise.filter(Column(Columns.BG_start_time.rawValue)==time).fetchOne(db)
        }
    }
    
    /// 查询某天dateStr 记录
    private static func queryRecords(dateStr: String) -> [YFRecordExercise] {
        guard let _dbQueue = dbQueue else { return [] }
        /// 创建数据库
        self.createTable()
        /// 返回查询结果
        return try! _dbQueue.unsafeRead({ (db) -> [YFRecordExercise] in
            return try YFRecordExercise.filter(Column(Columns.BG_dateStr.rawValue) == dateStr).order(Columns.BG_start_time).fetchAll(db)
        })
    }
    
    /// 查询某天 dateStr,  最近 limit 条记录
    private static func queryRecords(dateStr: String, limit: Int = Int.max) -> [YFRecordExercise] {
        guard let _dbQueue = dbQueue else { return [] }
        /// 创建数据库
        self.createTable()
        /// 返回查询结果
        return try! _dbQueue.unsafeRead({ (db) -> [YFRecordExercise] in
            return try YFRecordExercise.filter(Column(Columns.BG_dateStr.rawValue) == dateStr).order(Columns.BG_start_time).reversed().limit(limit).fetchAll(db)
        })
    }
    
    /// 查询 开始于时间错 fromTime 的记录
    private static func queryRecords(start fromTime: TimeInterval) -> [YFRecordExercise] {
        guard let _dbQueue = dbQueue else { return [] }
        /// 创建数据库
        self.createTable()
        /// 返回查询结果
        return try! _dbQueue.unsafeRead({ (db) -> [YFRecordExercise] in
            return try YFRecordExercise.filter(Column(Columns.BG_start_time.rawValue) >= fromTime).order(Columns.BG_start_time).fetchAll(db)
        })
    }
    
    /// 查询时间戳 【fromTime，toTime 】 的记录
    private static func queryRecords(start fromTime: TimeInterval, end toTime: TimeInterval) -> [YFRecordExercise] {
        guard let _dbQueue = dbQueue else { return [] }
        /// 创建数据库
        self.createTable()
        /// 返回查询结果
        return try! _dbQueue.unsafeRead({ (db) -> [YFRecordExercise] in
            return try YFRecordExercise.filter(Column(Columns.BG_start_time.rawValue) >= fromTime && Column(Columns.BG_start_time.rawValue) <= toTime).order(Columns.BG_start_time).fetchAll(db)
        })
    }
    
}


///: -  外部查询
extension YFRecordExercise {
    
    internal static var latestRecord: YFRecordExercise? {
        if let item = _latestRecord { return item }
        if let record =  queryRecordLatest() {
            _latestRecord = record;
            return record
        }
        return nil
    }
    
    internal static func queryLatest(with date: Date) -> YFRecordExercise? {
        let dateStr = self.dateStr(date)
        let records = queryRecords(dateStr: dateStr, limit: 1)
        return records.last
    }
    
    internal static func query(with date: Date) -> [YFRecordExercise] {
        let dateStr = self.dateStr(date)
        let records = queryRecords(dateStr: dateStr)
        return records
    }
    
    internal static func query(start fromTime: TimeInterval) -> [YFRecordExercise] {
        let records = queryRecords(start: fromTime)
        return records
    }
    
    internal static func query(start fromTime: TimeInterval, end toTime: TimeInterval) -> [YFRecordExercise] {
        let records = queryRecords(start: fromTime, end: toTime)
        return records
    }
}
