//
//  YFRecordHeart.swift
//  YFitKit
//
//  Created by Mac on 2021/5/12.
//

import Foundation
import GRDB

struct YFRecordHeart: Codable {
    var BG_bg_id: Int?
    var BG_bg_createTime: String = ""
    var BG_bg_updateTime: String = ""

    var BG_dateStr: String = ""
    var BG_time: TimeInterval = 0
    var BG_heartNum: UInt8 = 0
        
    init(with model: YFHeart, isUpdate:Bool = false) {
        self.BG_dateStr = model.dateStr
        self.BG_time = model.time
        self.BG_heartNum = model.heartNum

        let nowStr = YFCalendarUtils.datetimeStr(date: Date())
        if !isUpdate { self.BG_bg_createTime = nowStr }
        self.BG_bg_updateTime = nowStr
    }
}

// MARK: - ColumnExpression
extension YFRecordHeart {
    enum Columns: String, ColumnExpression {
        case BG_bg_id, BG_bg_createTime, BG_bg_updateTime
        case BG_dateStr, BG_time
        case BG_heartNum
    }
}

// MARK: - TableRecord
/// 使用 TableRecord 协议, 告知操作数据表
extension YFRecordHeart: TableRecord {
    //  数据表名字 如果不自定义 将使用默认的表名, 默认为小写开头驼峰命名 例如 UserEntity -> userEntity
    public static var databaseTableName: String { return "BHYHeartInfoHistory" }
    // 声明列名表示的方式
    public static var databaseSelection: [SQLSelectable] = [Columns.BG_bg_id, Columns.BG_bg_createTime, Columns.BG_bg_updateTime,
                                                            Columns.BG_dateStr, Columns.BG_time,
                                                            Columns.BG_heartNum
                                                            ]
}

// MARK: - FetchableRecord
/// FetchableRecord 进行查询操作
/// 使用了Codable可以不实现  init (row: Row)
/// 未使用 Codable = Decodable & Encodable 协议的需要实现 init (row: Row)
extension YFRecordHeart: FetchableRecord {
    public init(row: Row) {
        BG_bg_id = row[Columns.BG_bg_id]
        BG_bg_createTime = row[Columns.BG_bg_createTime]
        BG_bg_updateTime = row[Columns.BG_bg_updateTime]
        
        BG_dateStr = row[Columns.BG_dateStr]
        BG_time = row[Columns.BG_time]
        BG_heartNum = row[Columns.BG_heartNum]
    }
}

// MARK: - MutablePersistableRecord
/// 使用PersistableRecord / MutablePersistableRecord插入更新保存数据,
/// 存储模型是Class使用PersistableRecord,
/// 存储模型是struct使用MutablePersistableRecord.
/// 两者区别在于 MutablePersistableRecord save() insert() 是 mutating.
extension YFRecordHeart: MutablePersistableRecord {
    
    public func encode(to container: inout PersistenceContainer) {
        container[Columns.BG_bg_id] = BG_bg_id
        container[Columns.BG_bg_createTime] = BG_bg_createTime
        container[Columns.BG_bg_updateTime] = BG_bg_updateTime

        container[Columns.BG_dateStr] = BG_dateStr
        container[Columns.BG_time] = BG_time
        container[Columns.BG_heartNum] = BG_heartNum
    }
    
    mutating public func didInsert(with rowID: Int64, for column: String?) {
        
    }
    
}

extension YFRecordHeart {
    
    /// 获取数据库对象
    private static var dbQueue: DatabaseQueue? {
        SQLiteManager.dbQueue
    }
    
    /// 创建数据库
    private static func createTable() -> Void {
        guard let _dbQueue = dbQueue else { return }
        try! _dbQueue.inDatabase { (db) -> Void in
            /// 判断是否存在数据库
            if try db.tableExists(databaseTableName) { return }
            
            /// 创建数据库表
            try db.create(table: databaseTableName, temporary: false, ifNotExists: true, body: { (t) in
                t.autoIncrementedPrimaryKey(Columns.BG_bg_id.rawValue)
                t.column(Columns.BG_bg_createTime.rawValue, .text)
                t.column(Columns.BG_bg_updateTime.rawValue, .text)
                
                t.column(Columns.BG_dateStr.rawValue,.integer)
                t.column(Columns.BG_time.rawValue,.double)
                t.column(Columns.BG_heartNum.rawValue,.integer)
                
                t.uniqueKey([Columns.BG_time.rawValue])
            })
        }
    }
}


// MARK: - 插入&更新

extension YFRecordHeart {
    
    private static var now: Date { Date() }
    
    private static var _latestRecord: YFRecordHeart?
    
    /// 查询
    private static func queryRecordLatest() -> YFRecordHeart? {
        let records = queryAllRecord(limit: 1)
        return records.first
    }
    
    /// 更新
    private static func update(_ record: YFRecordHeart, _ item: YFHeart) {
        guard let _dbQueue = dbQueue else { return }
        /// 创建数据库表
        self.createTable()
        /// 事务 更新场景
        try! _dbQueue.inTransaction { (db) -> Database.TransactionCompletion in
            do {
                var temp = record
                temp.BG_heartNum = item.heartNum
                temp.BG_bg_updateTime = YFCalendarUtils.datetimeStr(date: Date())
                try temp.update(db)
                return Database.TransactionCompletion.commit
            } catch {
                return Database.TransactionCompletion.rollback
            }
        }
    }
    
    /// 插入
    private static func insert(_ item: YFHeart) {
        guard let _dbQueue = dbQueue else { return }
        /// 创建数据库表
        self.createTable()
        let record = YFRecordHeart(with: item, isUpdate: false)
        /// 事务
        try! _dbQueue.inTransaction { (db) -> Database.TransactionCompletion in
            do {
                var temp = record
                try temp.insert(db)
                return Database.TransactionCompletion.commit
            } catch {
                return Database.TransactionCompletion.rollback
            }
        }
    }
    
    /// 保存单个数据， dateStr 相等则更新， 否则插入
    @discardableResult
    internal static func save(with item: YFHeart) -> Bool {
        guard item.time > YFCalendarUtils.date(from: "2010", format: "yyyy").timeIntervalSince1970 else { return false}
        guard item.time < now.timeIntervalSince1970 + 10 else { return false}
        if let record =  latestRecord,  item.time < record.BG_time { return false}
        _latestRecord = YFRecordHeart(with: item)
        
        /// 判断是否存在
        if let record = queryRecord(time: item.time) {
            self.update(record, item)
        } else {
            self.insert(item)
        }
        return true
    }
}

// MARK: - 删除

extension YFRecordHeart {
    
    internal static func remove() {
        guard let _dbQueue = dbQueue else { return }
        /// 事务
        try! _dbQueue.inTransaction { (db) -> Database.TransactionCompletion in
            do {
                try db.drop(table: databaseTableName)
                return Database.TransactionCompletion.commit
            } catch {
                return Database.TransactionCompletion.rollback
            }
        }
    }
    
    /// 删除所有
    internal static func deleteAll() {
        _latestRecord = nil
        guard let _dbQueue = dbQueue else { return }
        /// 是否有数据库表
        self.createTable()
        /// 事务
        try! _dbQueue.inTransaction { (db) -> Database.TransactionCompletion in
            do {
                try YFRecordHeart.deleteAll(db)
                return Database.TransactionCompletion.commit
            } catch {
                return Database.TransactionCompletion.rollback
            }
        }
    }
}


// MARK: - 查询

extension YFRecordHeart {
    
    private static func dateStr(_ date: Date) -> String {
        return YFCalendarUtils.dateStr(date: date)
    }
    
    /// 查询指定时间戳
    private static func queryRecord(time: TimeInterval) -> YFRecordHeart? {
        guard let _dbQueue = dbQueue else { return nil }
        /// 创建数据库
        self.createTable()
        /// 返回查询结果
        return try! _dbQueue.unsafeRead { (db) -> YFRecordHeart? in
            return try YFRecordHeart.filter(Column(Columns.BG_time.rawValue)==time).fetchOne(db)
        }
    }
    
    /// 查询所有: 最近limit条记录
    private static func queryAllRecord(limit: Int = Int.max) -> [YFRecordHeart] {
        guard let _dbQueue = dbQueue else { return [] }
        /// 创建数据库
        self.createTable()
        /// 返回查询结果
        return try! _dbQueue.unsafeRead({ (db) -> [YFRecordHeart] in
            return try YFRecordHeart.order(Columns.BG_time).reversed().limit(limit).fetchAll(db)
        })
    }
    
    /// 查询某天dateStr 记录
    private static func queryRecords(dateStr: String) -> [YFRecordHeart] {
        guard let _dbQueue = dbQueue else { return [] }
        /// 创建数据库
        self.createTable()
        /// 返回查询结果
        return try! _dbQueue.unsafeRead({ (db) -> [YFRecordHeart] in
            return try YFRecordHeart.filter(Column(Columns.BG_dateStr.rawValue) == dateStr).order(Columns.BG_time).fetchAll(db)
        })
    }
    
    /// 查询某天 dateStr,  最近 limit 条记录
    private static func queryRecords(dateStr: String, limit: Int = Int.max) -> [YFRecordHeart] {
        guard let _dbQueue = dbQueue else { return [] }
        /// 创建数据库
        self.createTable()
        /// 返回查询结果
        return try! _dbQueue.unsafeRead({ (db) -> [YFRecordHeart] in
            return try YFRecordHeart.filter(Column(Columns.BG_dateStr.rawValue) == dateStr).order(Columns.BG_time).reversed().limit(limit).fetchAll(db)
        })
    }
    
    /// 查询 开始于时间错 fromTime 的记录
    private static func queryRecords(start fromTime: TimeInterval) -> [YFRecordHeart] {
        guard let _dbQueue = dbQueue else { return [] }
        /// 创建数据库
        self.createTable()
        /// 返回查询结果
        return try! _dbQueue.unsafeRead({ (db) -> [YFRecordHeart] in
            return try YFRecordHeart.filter(Column(Columns.BG_time.rawValue) >= fromTime).order(Columns.BG_time).fetchAll(db)
        })
    }
    
    /// 查询时间戳 【fromTime，toTime 】 的记录
    private static func queryRecords(start fromTime: TimeInterval, end toTime: TimeInterval) -> [YFRecordHeart] {
        guard let _dbQueue = dbQueue else { return [] }
        /// 创建数据库
        self.createTable()
        /// 返回查询结果
        return try! _dbQueue.unsafeRead({ (db) -> [YFRecordHeart] in
            return try YFRecordHeart.filter(Column(Columns.BG_time.rawValue) >= fromTime && Column(Columns.BG_time.rawValue) <= toTime).order(Columns.BG_time).fetchAll(db)
        })
    }
    
}


///: -  外部查询
extension YFRecordHeart {
    
    internal static var latestRecord: YFRecordHeart? {
        if let item = _latestRecord { return item }
        if let record =  queryRecordLatest() {
            _latestRecord = record;
            return record
        }
        return nil
    }
    
    internal static func queryLatest(with date: Date) -> YFRecordHeart? {
        let dateStr = self.dateStr(date)
        let records = queryRecords(dateStr: dateStr, limit: 1)
        return records.last
    }
    
    internal static func query(with date: Date) -> [YFRecordHeart] {
        let dateStr = self.dateStr(date)
        let records = queryRecords(dateStr: dateStr)
        return records
    }
    
    internal static func query(start fromTime: TimeInterval) -> [YFRecordHeart] {
        let records = queryRecords(start: fromTime)
        return records
    }
    
    internal static func query(start fromTime: TimeInterval, end toTime: TimeInterval) -> [YFRecordHeart] {
        let records = queryRecords(start: fromTime, end: toTime)
        return records
    }
}
