//
//  YFRecordTotal.swift
//  YFitKit
//
//  Created by Mac on 2021/5/8.
//

import Foundation
import GRDB

struct YFRecordTotal: Codable {
    var BG_bg_id: Int?
    var BG_bg_createTime: String = ""
    var BG_bg_updateTime: String = ""

    var BG_dateStr: String = ""
    var BG_startSecs: TimeInterval = 0
    var BG_walkDuration: UInt32 = 0
    var BG_walkSteps: UInt32 = 0
    var BG_walkDistance: UInt32 = 0
    var BG_walkCalories: UInt32 = 0
        
    init(with model: YFTotal, isUpdate:Bool = false) {
        self.BG_dateStr = model.dateStr
        self.BG_startSecs = model.startSecs
        self.BG_walkDuration = model.walkDuration
        self.BG_walkSteps = model.walkSteps
        self.BG_walkDistance = model.walkDistance
        self.BG_walkCalories = model.walkCalories

        let nowStr = YFCalendarUtils.datetimeStr(date: Date())
        if !isUpdate { self.BG_bg_createTime = nowStr }
        self.BG_bg_updateTime = nowStr
    }
}

// MARK: - ColumnExpression
extension YFRecordTotal {
    enum Columns: String, ColumnExpression {
        case BG_bg_id, BG_bg_createTime, BG_bg_updateTime
        case BG_dateStr, BG_startSecs
        case BG_walkDuration, BG_walkSteps, BG_walkDistance, BG_walkCalories
    }
}

// MARK: - TableRecord
/// 使用 TableRecord 协议, 告知操作数据表
extension YFRecordTotal: TableRecord {
    //  数据表名字 如果不自定义 将使用默认的表名, 默认为小写开头驼峰命名 例如 UserEntity -> userEntity
    public static var databaseTableName: String { return "BHYSportTotal" }
    // 声明列名表示的方式
    public static var databaseSelection: [SQLSelectable] = [Columns.BG_bg_id, Columns.BG_bg_createTime, Columns.BG_bg_updateTime,
                                                            Columns.BG_dateStr, Columns.BG_startSecs,
                                                            Columns.BG_walkDuration,Columns.BG_walkSteps,
                                                            Columns.BG_walkDistance, Columns.BG_walkCalories,
                                                            ]
}


// MARK: - FetchableRecord
/// FetchableRecord 进行查询操作
/// 使用了Codable可以不实现  init (row: Row)
/// 未使用 Codable = Decodable & Encodable 协议的需要实现 init (row: Row)
extension YFRecordTotal: FetchableRecord {
    public init(row: Row) {
        BG_bg_id = row[Columns.BG_bg_id]
        BG_bg_createTime = row[Columns.BG_bg_createTime]
        BG_bg_updateTime = row[Columns.BG_bg_updateTime]
        
        BG_dateStr = row[Columns.BG_dateStr]
        BG_startSecs = row[Columns.BG_startSecs]
        BG_walkDuration = row[Columns.BG_walkDuration]
        BG_walkSteps = row[Columns.BG_walkSteps]
        BG_walkDistance = row[Columns.BG_walkDistance]
        BG_walkCalories = row[Columns.BG_walkCalories]
    }
}

// MARK: - MutablePersistableRecord
/// 使用PersistableRecord / MutablePersistableRecord插入更新保存数据,
/// 存储模型是Class使用PersistableRecord,
/// 存储模型是struct使用MutablePersistableRecord.
/// 两者区别在于 MutablePersistableRecord save() insert() 是 mutating.
extension YFRecordTotal: MutablePersistableRecord {
    
    public func encode(to container: inout PersistenceContainer) {
        container[Columns.BG_bg_id] = BG_bg_id
        container[Columns.BG_bg_createTime] = BG_bg_createTime
        container[Columns.BG_bg_updateTime] = BG_bg_updateTime

        container[Columns.BG_dateStr] = BG_dateStr
        container[Columns.BG_startSecs] = BG_startSecs
        container[Columns.BG_walkDuration] = BG_walkDuration
        container[Columns.BG_walkSteps] = BG_walkSteps
        container[Columns.BG_walkDistance] = BG_walkDistance
        container[Columns.BG_walkCalories] = BG_walkCalories
    }
    
    mutating public func didInsert(with rowID: Int64, for column: String?) {
        
    }
    
}

extension YFRecordTotal {
    
    /// 获取数据库对象
    private static var dbQueue: DatabaseQueue? {
        SQLiteManager.dbQueue
    }
    
    /// 创建数据库
    private static func createTable() -> Void {
        guard let _dbQueue = dbQueue else { return }
        try! _dbQueue.inDatabase { (db) -> Void in
            /// 判断是否存在数据库
            if try db.tableExists(databaseTableName) { return }
            
            /// 创建数据库表
            try db.create(table: databaseTableName, temporary: false, ifNotExists: true, body: { (t) in
                t.autoIncrementedPrimaryKey(Columns.BG_bg_id.rawValue)
                t.column(Columns.BG_bg_createTime.rawValue, .text)
                t.column(Columns.BG_bg_updateTime.rawValue, .text)
                
                t.column(Columns.BG_dateStr.rawValue,.integer)
                t.column(Columns.BG_startSecs.rawValue,.double)
                t.column(Columns.BG_walkDuration.rawValue,.integer)
                t.column(Columns.BG_walkSteps.rawValue,.integer)
                t.column(Columns.BG_walkDistance.rawValue,.integer)
                t.column(Columns.BG_walkCalories.rawValue,.integer)
                
                t.uniqueKey([Columns.BG_startSecs.rawValue])
            })
        }
    }
}

// MARK: - 插入&更新

extension YFRecordTotal {
    
    private static var now: Date { Date() }
    
    private static var _latestRecord: YFRecordTotal?
    
    /// 查询
    private static func queryRecordLatest() -> YFRecordTotal? {
        let records = queryAllRecord(limit: 1)
        return records.first
    }
    
    /// 更新
    private static func update(_ record: YFRecordTotal, _ item: YFTotal) {
        guard let _dbQueue = dbQueue else { return }
        /// 创建数据库表
        self.createTable()
        /// 事务 更新场景
        try! _dbQueue.inTransaction { (db) -> Database.TransactionCompletion in
            do {
                var temp = record
                temp.BG_walkDuration = item.walkDuration
                temp.BG_walkSteps    = item.walkSteps
                temp.BG_walkDistance = item.walkDistance
                temp.BG_walkCalories = item.walkCalories
                temp.BG_bg_updateTime = YFCalendarUtils.datetimeStr(date: Date())
                try temp.update(db)
                return Database.TransactionCompletion.commit
            } catch {
                return Database.TransactionCompletion.rollback
            }
        }
    }
    
    /// 插入
    private static func insert(_ item: YFTotal) {
        guard let _dbQueue = dbQueue else { return }
        /// 创建数据库表
        self.createTable()
        let record = YFRecordTotal(with: item, isUpdate: false)
        /// 事务
        try! _dbQueue.inTransaction { (db) -> Database.TransactionCompletion in
            do {
                var temp = record
                try temp.insert(db)
                return Database.TransactionCompletion.commit
            } catch {
                return Database.TransactionCompletion.rollback
            }
        }
    }
    
    
    ///: - 验证存储处理
    private static func vaild(_ lastItem: YFTotal, _ item: YFTotal) -> Bool {
        /// 判断可存储记录， 必须大于上次存储最后存储记录时间戳
        /// startSecs 相等可能有数值变化
        guard item.startSecs >= lastItem.startSecs else { return false}
        /// 同一天的情况下:  步数必须比上次存储的数值大
        /// clearDay: 用于处理不同处理方式
        /// true: 清除当天之前的数据，再存储新数据
        /// false: 不存储
        //let clearDay: Bool = true
        let someDay: Bool = YFCalendarUtils.someDay(from: lastItem.startSecs, to: item.startSecs)
        if someDay, item.walkSteps < lastItem.walkSteps {
            //if !clearDay { return false }
            deleteDay(item.startSecs)
        }
        return true
    }
    
    /// 保存单个数据， dateStr 相等则更新， 否则插入
    @discardableResult
    internal static func save(with item: YFTotal) -> Bool {
        guard item.startSecs > YFCalendarUtils.date(from: "2010", format: "yyyy").timeIntervalSince1970 else { return false}
        guard item.startSecs < now.timeIntervalSince1970 + 10 else { return false}
        if let record =  latestRecord {
            let lastItem = YFTotal.parse(with: record)
            guard vaild(lastItem, item) else { return false }
        }
        
        _latestRecord = YFRecordTotal(with: item)
        /// 判断是否存在
        if let record = queryRecord(time: item.startSecs) {
            self.update(record, item)
        } else {
            self.insert(item)
        }
        return true
    }
}


// MARK: - 删除

extension YFRecordTotal {
    
    ///: - 移除表
    internal static func remove() {
        _latestRecord = nil
        guard let _dbQueue = dbQueue else { return }
        /// 事务
        try! _dbQueue.inTransaction { (db) -> Database.TransactionCompletion in
            do {
                try db.drop(table: databaseTableName)
                return Database.TransactionCompletion.commit
            } catch {
                return Database.TransactionCompletion.rollback
            }
        }
    }
    
    ///: - 删除某天记录
    internal static func deleteDay(_ time: TimeInterval) {
        let dateStr = dateStr(Date(timeIntervalSince1970: time))
        _latestRecord = nil
        guard let _dbQueue = dbQueue else { return }
        /// 是否有数据库表
        self.createTable()
        /// 事务
        try! _dbQueue.inTransaction { (db) -> Database.TransactionCompletion in
            do {
                try YFRecordTotal.filter(Column(Columns.BG_dateStr.rawValue) == dateStr).deleteAll(db)
                return Database.TransactionCompletion.commit
            } catch {
                return Database.TransactionCompletion.rollback
            }
        }
    }
    
    ///: - 删除所有记录
    internal static func deleteAll() {
        _latestRecord = nil
        guard let _dbQueue = dbQueue else { return }
        /// 是否有数据库表
        self.createTable()
        /// 事务
        try! _dbQueue.inTransaction { (db) -> Database.TransactionCompletion in
            do {
                try YFRecordTotal.deleteAll(db)
                return Database.TransactionCompletion.commit
            } catch {
                return Database.TransactionCompletion.rollback
            }
        }
    }
}

// MARK: - 查询

extension YFRecordTotal {
    
    private static func dateStr(_ date: Date) -> String {
        return YFCalendarUtils.dateStr(date: date)
    }
    
    /// 查询指定时间戳
    private static func queryRecord(time : TimeInterval) -> YFRecordTotal? {
        guard let _dbQueue = dbQueue else { return nil }
        /// 创建数据库
        self.createTable()
        /// 返回查询结果
        return try! _dbQueue.unsafeRead { (db) -> YFRecordTotal? in
            return try YFRecordTotal.filter(Column(Columns.BG_startSecs.rawValue)==time).fetchOne(db)
        }
    }
    
    /// 查询所有: 最近limit条记录
    private static func queryAllRecord(limit: Int = Int.max) -> [YFRecordTotal] {
        guard let _dbQueue = dbQueue else { return [] }
        /// 创建数据库
        self.createTable()
        /// 返回查询结果
        return try! _dbQueue.unsafeRead({ (db) -> [YFRecordTotal] in
            return try YFRecordTotal.order(Columns.BG_startSecs).reversed().limit(limit).fetchAll(db)
        })
    }
    
    /// 查询某天 dateStr 记录
    private static func queryRecords(dateStr: String) -> [YFRecordTotal] {
        guard let _dbQueue = dbQueue else { return [] }
        /// 创建数据库
        self.createTable()
        /// 返回查询结果
        return try! _dbQueue.unsafeRead({ (db) -> [YFRecordTotal] in
            return try YFRecordTotal.filter(Column(Columns.BG_dateStr.rawValue) == dateStr).order(Columns.BG_startSecs).fetchAll(db)
        })
    }
    
    /// 查询某天 dateStr,  最近 limit 条记录
    private static func queryRecords(dateStr: String, limit: Int = Int.max) -> [YFRecordTotal] {
        guard let _dbQueue = dbQueue else { return [] }
        /// 创建数据库
        self.createTable()
        /// 返回查询结果
        return try! _dbQueue.unsafeRead({ (db) -> [YFRecordTotal] in
            return try YFRecordTotal.filter(Column(Columns.BG_dateStr.rawValue) == dateStr).order(Columns.BG_startSecs).reversed().limit(limit).fetchAll(db)
        })
    }
    
    /// 查询 开始于时间戳 fromTime 的记录
    private static func queryRecords(start fromTime: TimeInterval) -> [YFRecordTotal] {
        guard let _dbQueue = dbQueue else { return [] }
        /// 创建数据库
        self.createTable()
        /// 返回查询结果
        return try! _dbQueue.unsafeRead({ (db) -> [YFRecordTotal] in
            return try YFRecordTotal.filter(Column(Columns.BG_startSecs.rawValue) >= fromTime).order(Columns.BG_startSecs).fetchAll(db)
        })
    }
}


/// 外部查询
extension YFRecordTotal {
    
    internal static var latestRecord: YFRecordTotal? {
        if let item = _latestRecord { return item }
        if let record =  queryRecordLatest() {
            _latestRecord = record;
            return record
        }
        return nil
    }
    
    internal static func queryLatest(with date: Date) -> YFRecordTotal? {
        let dateStr = self.dateStr(date)
        let records = queryRecords(dateStr: dateStr, limit: 1)
        return records.last
    }
    
    /// 查询某时间戳之前最近的一条记录
    internal static func queryLatest(before time: TimeInterval) -> YFRecordTotal? {
        guard let _dbQueue = dbQueue else { return nil }
        /// 创建数据库
        self.createTable()
        /// 返回查询结果
        return try! _dbQueue.unsafeRead({ (db) -> YFRecordTotal? in
            return try YFRecordTotal.filter(Column(Columns.BG_startSecs.rawValue) < time).order(Columns.BG_startSecs).reversed().limit(1).fetchAll(db).last
            
        })
    }
    
    internal static func query(with date: Date) -> [YFRecordTotal] {
        let dateStr = self.dateStr(date)
        let records = queryRecords(dateStr: dateStr)
        return records
    }
    
    internal static func query(start fromTime: TimeInterval) -> [YFRecordTotal] {
        queryRecords(start: fromTime)
    }
}
