//
//  YFKitService.swift
//  YFitKit
//
//  Created by Mac on 2021/1/31.
//

import Foundation
import YFitSDK
import RxSwift

enum YFReponseError: Int {
    case sendFailed = -2000
    case overTime
    case bleClosed
    case deviceDisconnect
    case bleExceptional
    
    static let nohandler_collection: [YFReponseError] = [.bleClosed, .deviceDisconnect, .bleExceptional]
    static func invaild(_ error: Error?) -> Bool {
        if let err = error,
           let hasError = YFReponseError(rawValue: err._code),
           nohandler_collection.contains(hasError) { return true}
        return false
    }
}

public class YFKitService {
    
    /// : - Subject
    private let foundDeviceSubject = PublishSubject<[YFPeripheral]>()
    private let productStatusSubject = PublishSubject<YFProductStatus>()
    private let receiveDatasSubject = PublishSubject<Dictionary<String, Any>>()
    
    private init() {
        observerBandNotifys()
        DispatchQueue.main.async { _ = YFDataManager.shared }
        DispatchQueue.main.async { _ = self.product }
    }
    
    private let product: CEProductK6 = CEProductK6.shareInstance()
    
    private var uuidForBindPeripheral: String? {
        guard let uuid = product.lastConnectUUId else { return nil }
        return uuid
    }
    
    var currentPeripheral: YFPeripheral?

    // MARK: - Fileprivate
    /// : - 绑定

    fileprivate var hasBind: Bool {
        guard let _ = product.lastConnectUUId else { return false}
        return true
    }
    fileprivate func bind() { product.saveConnectedUUid() }
    fileprivate func unbind() { clearReaded(); product.releaseBind() }

    ///: - 清除已读设置记录标志
    fileprivate func clearReaded() {
        YFHybridInfo.readed = false
        YFClockInfo.readed  = false
    }
    
    /// : - 搜索设备
    fileprivate func searchDeviceStart() {
        guard product.status != .powerOff else { return }
        product.startScan()
    }

    fileprivate func searchDeviceStop() {
        guard product.status != .powerOff else { return }
        product.connect.stopScan()
    }
   
    /// : - 连接设备
    fileprivate func connect(for peripheral: YFPeripheral) {
        product.connect(peripheral.peripheral)
    }
    fileprivate func disconnect() { product.breakConnect() }

    
    /// : - 发送指令
    fileprivate func sendCmdToDevice(cmd: CE_Cmd, closure: @escaping (Error?) -> Void)  {
        // guard product.status == .completed else {  return }
        product.sendCmd(toDevice: cmd) { [weak self] (error) in
            guard let strong = self else { return }
            if YFReponseError.invaild(error) { strong.product.cleanCmdQueue() }
            closure(error)
        }
    }
}


// MARK: - observer & selector
extension YFKitService {
    
    private func observerBandNotifys() {
        NotificationCenter.default.addObserver(self, selector: #selector(foundDevice),
                                               name: NSNotification.Name(rawValue: ScanPeripheralsNoticeKey), object: nil)
        NotificationCenter.default.addObserver(self, selector: #selector(productStatusChanged),
                                               name: NSNotification.Name(rawValue: ProductStatusChangeNoticeKey), object: nil)
        NotificationCenter.default.addObserver(self, selector: #selector(receiveData),
                                               name: NSNotification.Name(rawValue: CEProductK6ReceiveDataNoticeKey), object: nil)
    }
    /// 发现设备
    @objc private func foundDevice(_ notify: Notification) {
        guard let peripherals = notify.object as? [SearchPeripheral] else { return }
        foundDeviceSubject.onNext(peripherals.map{ YFPeripheral($0) })
    }
    /// 状态更改
    @objc private func productStatusChanged(_ notify: Notification) {
        guard let value = notify.object  as? Int,
              let status = YFProductStatus.parse(value) else { return }
        YFLog.i(status.text)
        
        switch status {
        case .connected:
            if let peripheral = product.connect.connectingPeripheral {
                currentPeripheral = YFPeripheral(peripheral)
            }
        case .completed: break

        default: currentPeripheral = nil
        }

        productStatusSubject.onNext(status)
    }
    
    /// 返回数据
    @objc private func receiveData(_ notify: Notification) {
        guard let userInfo = notify.userInfo as? Dictionary<String, Any> else { return }
        receiveDatasSubject.onNext(userInfo)
    }
}


// MARK: - Class methods

extension YFKitService {
    
    private static let shared: YFKitService = YFKitService()
    
    public static var isConnected: Bool {
        (shared.product.status == .connected || shared.product.status == .completed)
    }
    public static var autoConnecting: Bool {
        get { shared.product.isAutoConnect }
        set { shared.product.isAutoConnect = newValue }
    }
    
    public static var currentPeripheral: YFPeripheral? { shared.currentPeripheral }

    
    /// : - Subject - Public output
    public static var foundDeviceOutput: Observable<[YFPeripheral]> {
        return shared.foundDeviceSubject.share(replay: 1, scope: .forever).asObservable()
    }
    public static var productStatusOutput: Observable<YFProductStatus> {
        return shared.productStatusSubject.asObservable()
    }
    public static var receiveDatasOutput: Observable<Dictionary<String, Any>> {
        return shared.receiveDatasSubject.asObservable()
    }

    /// : - 搜索设备
    public static func searchDeviceStart() { shared.searchDeviceStart()  }
    public static func searchDeviceStop() { shared.searchDeviceStop() }
   
    /// : - 连接设备
    public static func connect(for peripheral: YFPeripheral) { shared.connect(for: peripheral) }
    public static func disconnect() { shared.disconnect() }
    public static func reconnectBindPeripheral() { shared.product.startAutoConnect() }

    
    /// : - 绑定关系
    public static var  binded: Bool { shared.hasBind }
    public static func bind() { shared.bind() }
    public static func unbind() { shared.unbind() }
    
    /// : - 发送指令
    public static func sendCmdToDevice(cmd: CE_Cmd, closure: @escaping (Error?) -> Void)  {
        shared.sendCmdToDevice(cmd: cmd, closure: closure)
    }
}

extension YFKitService {
    /// :- 该设备有改变
    /// 1. 天气推送功能
    /// 2. 中文替换为德文
    static var isSW294: Bool {
        guard let name = currentPeripheral?.name,
              name == "SW294" else { return false }
        return true
    }
}
