//
//  YFPeripheral.swift
//  YFitKit
//
//  Created by Mac on 2021/4/20.
//

import Foundation
import CoreBluetooth
import YFitSDK

/// 表示被搜索到的外设实例
public class YFPeripheral {
    
    // MARK: - Public
    public lazy var peripheralName : String = {
        var devname: String?
        if let hasName = peripheral.name { devname = hasName }
        if let hasName = name { devname = hasName }
        var devid: String = ""
        if let hasDevid = deviceID {  devid = "-\(hasDevid)"}
        if let name = devname { return name + devid }
        return "Unknown Name"
    } ()
    
    public var rssiText : String { return "rssi: \(rssi)" }
    
    public var rssiValue : Int { return rssi.intValue }
    
    public var uuid: String { return peripheral.identifier.uuidString }

    // MARK: Init
    init(_ peripheral: SearchPeripheral) {
        self.peripheral = peripheral.peripheral!.copy() as! CBPeripheral
        self.advertisementData = peripheral.advertisementData
        self.rssi = peripheral.rssi ?? NSNumber(value: 0)
        self.name = peripheral.name()
        self.deviceID = peripheral.deviceID()
        self.version = peripheral.version()
    }
    
    public init(_ peripheral: CBPeripheral) {
        self.peripheral = peripheral
        self.advertisementData = nil
        self.name = peripheral.name
        self.deviceID = nil
        self.version = 0
    }
    
    // MARK: - Internal
    /// `Peripheral` 实例, 这允许执行进一步的蓝牙操作.
    let peripheral: CBPeripheral
    /// 扫描peripheral的广告数据
    let advertisementData: Dictionary<AnyHashable, Any>?
    /// RSSI value.
    var rssi: NSNumber = NSNumber(value: 0)
    
    /// 下面数据由广播数据解析出来
    /// 设备名称
    let name: String?
    /// 展示在蓝牙搜索里面的四位ID
    let deviceID: String?
    /// 协议版本，根据此版本判断使用哪个版本的协议
    var version: Int = 0
    
}

extension YFPeripheral: Comparable {

    public static func < (lhs: YFPeripheral, rhs: YFPeripheral) -> Bool {
        return (lhs.rssi.intValue < rhs.rssi.intValue)
    }
    
    public static func == (lhs: YFPeripheral, rhs: YFPeripheral) -> Bool {
        return lhs.peripheral.identifier.uuidString == rhs.peripheral.identifier.uuidString
    }
}

extension YFPeripheral: CustomStringConvertible {
    public var description: String {
        return "\(name ?? "--")-\(deviceID ?? "XXXX")   \(peripheral)"
    }
}
