/*
* Copyright (c) 2019, Nordic Semiconductor
* All rights reserved.
*
* Redistribution and use in source and binary forms, with or without modification,
* are permitted provided that the following conditions are met:
*
* 1. Redistributions of source code must retain the above copyright notice, this
*    list of conditions and the following disclaimer.
*
* 2. Redistributions in binary form must reproduce the above copyright notice, this
*    list of conditions and the following disclaimer in the documentation and/or
*    other materials provided with the distribution.
*
* 3. Neither the name of the copyright holder nor the names of its contributors may
*    be used to endorse or promote products derived from this software without
*    specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
* ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
* IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
* INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
* NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
* PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
* WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
* ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
* POSSIBILITY OF SUCH DAMAGE.
*/

import CoreBluetooth

internal enum DFUOpCode : UInt8 {
    case startDfu                           = 1
    case initDfuParameters                  = 2
    case receiveFirmwareImage               = 3
    case validateFirmware                   = 4
    case activateAndReset                   = 5
    case reset                              = 6
    case reportReceivedImageSize            = 7 // unused in this library
    case packetReceiptNotificationRequest   = 8
    case responseCode                       = 16
    case packetReceiptNotification          = 17
    
    var code: UInt8 {
        return rawValue
    }
}

internal enum InitDfuParametersRequest : UInt8 {
    case receiveInitPacket  = 0
    case initPacketComplete = 1
    
    var code: UInt8 {
        return rawValue
    }
}

internal enum Request {
    case jumpToBootloader
    case startDfu(type: UInt8)
    case startDfu_v1
    case initDfuParameters(req: InitDfuParametersRequest)
    case initDfuParameters_v1
    case receiveFirmwareImage
    case validateFirmware
    case activateAndReset
    case reset
    case packetReceiptNotificationRequest(number: UInt16)
    
    var data : Data {
        switch self {
        case .jumpToBootloader:
            return Data([DFUOpCode.startDfu.code, FIRMWARE_TYPE_APPLICATION])
        case .startDfu(let type):
            return Data([DFUOpCode.startDfu.code, type])
        case .startDfu_v1:
            return Data([DFUOpCode.startDfu.code])
        case .initDfuParameters(let req):
            return Data([DFUOpCode.initDfuParameters.code, req.code])
        case .initDfuParameters_v1:
            return Data([DFUOpCode.initDfuParameters.code])
        case .receiveFirmwareImage:
            return Data([DFUOpCode.receiveFirmwareImage.code])
        case .validateFirmware:
            return Data([DFUOpCode.validateFirmware.code])
        case .activateAndReset:
            return Data([DFUOpCode.activateAndReset.code])
        case .reset:
            return Data([DFUOpCode.reset.code])
        case .packetReceiptNotificationRequest(let number):
            var data = Data([DFUOpCode.packetReceiptNotificationRequest.code])
            data += number.littleEndian
            return data
        }
    }
    
    var description : String {
        switch self {
        case .jumpToBootloader:     return "Jump to bootloader (Op Code = 1, Upload Mode = 4)"
        case .startDfu(let type):   return "Start DFU (Op Code = 1, Upload Mode = \(type))"
        case .startDfu_v1:          return "Start DFU (Op Code = 1)"
        case .initDfuParameters(_): return "Initialize DFU Parameters"
        case .initDfuParameters_v1: return "Initialize DFU Parameters"
        case .receiveFirmwareImage: return "Receive Firmware Image (Op Code = 3)"
        case .validateFirmware:     return "Validate Firmware (Op Code = 4)"
        case .activateAndReset:     return "Activate and Reset (Op Code = 5)"
        case .reset:                return "Reset (Op Code = 6)"
        case .packetReceiptNotificationRequest(let number):
                                    return "Packet Receipt Notif Req (Op Code = 8, Value = \(number))"
        }
    }
}

internal enum DFUResultCode : UInt8 {
    case success              = 1
    case invalidState         = 2
    case notSupported         = 3
    case dataSizeExceedsLimit = 4
    case crcError             = 5
    case operationFailed      = 6
    
    // Note: When more result codes are added, the corresponding DFUError
    //       case needs to be added. See `error` property below.
    
    var code: UInt8 {
        return rawValue
    }
    
    var error: DFUError {
        return DFURemoteError.legacy.with(code: code)
    }
    
    var description: String {
        switch self {
        case .success:              return "Success"
        case .invalidState:         return "Device is in invalid state"
        case .notSupported:         return "Operation not supported"
        case .dataSizeExceedsLimit: return "Data size exceeds limit"
        case .crcError:             return "CRC Error"
        case .operationFailed:      return "Operation failed"
        }
    }
}

internal struct Response {
    let opCode        : DFUOpCode
    let requestOpCode : DFUOpCode
    let status        : DFUResultCode
    
    init?(_ data: Data) {
        guard data.count >= 3,
              let opCode = DFUOpCode(rawValue: data[0]),
              let requestOpCode = DFUOpCode(rawValue: data[1]),
              let status = DFUResultCode(rawValue: data[2]),
              opCode == .responseCode else {
            return nil
        }
        
        self.opCode        = opCode
        self.requestOpCode = requestOpCode
        self.status        = status
    }
    
    var description: String {
        return "Response (Op Code = \(requestOpCode.rawValue), Status = \(status.rawValue))"
    }
}

internal struct PacketReceiptNotification {
    let opCode        : DFUOpCode
    let bytesReceived : UInt32
    
    init?(_ data: Data) {
        guard data.count >= 5,
              let opCode = DFUOpCode(rawValue: data[0]),
              opCode == .packetReceiptNotification else {
            return nil
        }
        
        self.opCode = opCode
        
        // According to https://github.com/NordicSemiconductor/IOS-Pods-DFU-Library/issues/54
        // in SDK 5.2.0.39364 the `bytesReveived` value in a PRN packet is 16-bit long,
        // instad of 32-bit. However, the packet is still 5 bytes long and the two last
        // bytes are 0x00-00. This has to be taken under consideration when comparing
        // number of bytes sent and received as the latter counter may rewind if fw size
        // is > 0xFFFF bytes (LegacyDFUService:L543).
        self.bytesReceived = data.asValue(offset: 1)
    }
}

@objc internal class DFUControlPoint : NSObject, CBPeripheralDelegate, DFUCharacteristic {

    internal var characteristic: CBCharacteristic
    internal var logger: LoggerHelper

    private var success: Callback?
    private var proceed: ProgressCallback?
    private var report:  ErrorCallback?
    private var request: Request?
    private var uploadStartTime: CFAbsoluteTime?
    private var resetSent = false
    
    internal var valid: Bool {
        return characteristic.properties.isSuperset(of: [.write, .notify])
    }
    
    // MARK: - Initialization

    required init(_ characteristic: CBCharacteristic, _ logger: LoggerHelper) {
        self.characteristic = characteristic
        self.logger = logger
    }

    // MARK: - Characteristic API methods
    
    /**
     Enables notifications for the DFU Control Point characteristics.
     Reports success or an error using callbacks.
    
     - parameter success: Method called when notifications were successfully enabled.
     - parameter report:  Method called in case of an error.
     */
    func enableNotifications(onSuccess success: Callback?, onError report: ErrorCallback?) {
        // Get the peripheral object.
        let optService: CBService? = characteristic.service
        guard let peripheral = optService?.peripheral else {
            report?(.invalidInternalState, "Assert characteristic.service?.peripheral != nil failed")
            return
        }
        
        // Save callbacks.
        self.success = success
        self.report  = report
        
        // Set the peripheral delegate to self.
        peripheral.delegate = self
        
        logger.v("Enabling notifications for \(characteristic.uuid.uuidString)...")
        logger.d("peripheral.setNotifyValue(true, for: \(characteristic.uuid.uuidString))")
        peripheral.setNotifyValue(true, for: characteristic)
    }
    
    /**
     Sends given request to the DFU Control Point characteristic.
     Reports success or an error using callbacks.
     
     - parameter request: Request to be sent.
     - parameter success: Method called when peripheral reported with status success.
     - parameter report:  Method called in case of an error.
     */
    func send(_ request: Request, onSuccess success: Callback?, onError report: ErrorCallback?) {
        // Get the peripheral object.
        let optService: CBService? = characteristic.service
        guard let peripheral = optService?.peripheral else {
            report?(.invalidInternalState, "Assert characteristic.service?.peripheral != nil failed")
            return
        }
        
        // Save callbacks and parameter.
        self.success   = success
        self.report    = report
        self.request   = request
        self.resetSent = false
        
        // Set the peripheral delegate to self.
        peripheral.delegate = self
        
        switch request {
        case .initDfuParameters(let req):
            if req == InitDfuParametersRequest.receiveInitPacket {
                logger.a("Writing \(request.description)...")
            }
        case .initDfuParameters_v1:
            logger.a("Writing \(request.description)...")
        case .jumpToBootloader, .activateAndReset, .reset:
            // Those three requests may not be confirmed by the remote DFU target.
            // The device may be restarted before sending the ACK.
            // This would cause an error in `peripheral:didWriteValueForCharacteristic:error`,
            // which can be ignored in this case.
            resetSent = true
        default:
            break
        }
        logger.v("Writing to characteristic \(characteristic.uuid.uuidString)...")
        logger.d("peripheral.writeValue(0x\(request.data.hexString), for: \(characteristic.uuid.uuidString), type: .withResponse)")
        peripheral.writeValue(request.data, for: characteristic, type: .withResponse)
    }
    
    /**
     Sets the callbacks used later on when a Packet Receipt Notification is received,
     a device reported an error or the whole firmware has been sent and the notification
     with success status was received. Sending the firmware is done using DFU Packet
     characteristic.
     
     - parameter success: Method called when peripheral reported with status success.
     - parameter proceed: Method called the a PRN has been received and sending following
     data can be resumed.
     - parameter report:  Method called in case of an error.
     */
    func waitUntilUploadComplete(onSuccess success: Callback?,
                                 onPacketReceiptNofitication proceed: ProgressCallback?,
                                 onError report: ErrorCallback?) {
        // Get the peripheral object.
        let optService: CBService? = characteristic.service
        guard let peripheral = optService?.peripheral else {
            report?(.invalidInternalState, "Assert characteristic.service?.peripheral != nil failed")
            return
        }
        
        // Save callbacks. The proceed callback will be called periodically whenever
        // a packet receipt notification is received. It resumes uploading.
        self.success = success
        self.proceed = proceed
        self.report  = report
        self.uploadStartTime = CFAbsoluteTimeGetCurrent()
        
        // Set the peripheral delegate to self.
        peripheral.delegate = self
    }
    
    // MARK: - Peripheral Delegate callbacks
    
    func peripheral(_ peripheral: CBPeripheral,
                    didUpdateNotificationStateFor characteristic: CBCharacteristic,
                    error: Error?) {
        if let error = error {
            logger.e("Enabling notifications failed. Check if Service Changed service is enabled.")
            logger.e(error)
            // Note:
            // Error 253: Unknown ATT error.
            // This most proably is a caching issue. Check if your device had
            // Service Changed characteristic (for non-bonded devices) in both
            // app and bootloader modes. For bonded devices make sure it sends
            // the Service Changed indication after connecting.
            report?(.enablingControlPointFailed, "Enabling notifications failed")
            return
        }
        
        logger.v("Notifications enabled for \(characteristic.uuid.uuidString)")
        logger.a("DFU Control Point notifications enabled")
        success?()
    }
    
    func peripheral(_ peripheral: CBPeripheral,
                    didWriteValueFor characteristic: CBCharacteristic,
                    error: Error?) {
        // This method, according to the iOS documentation, should be called
        // only after writing with response to a characteristic. However, on
        // iOS 10 this method is called even after writing without response,
        // which is a bug. The DFU Control Point characteristic always writes
        // with response, in oppose to the DFU Packet, which uses write without
        // response.
        guard self.characteristic.isEqual(characteristic) else {
            return
        }
        guard let request = request else {
            return
        }

        if let error = error {
            if !resetSent {
                logger.e("Writing to characteristic failed. Check if Service Changed characteristic is enabled.")
                logger.e(error)
                // Note:
                // Error 3: Writing is not permitted
                // This most proably is caching issue. Check if your device had
                // Service Changed characteristic (for non-bonded devices) in both
                // app and bootloader modes. For bonded devices make sure it sends
                // the Service Changed indication after connecting.
                report?(.writingCharacteristicFailed, "Writing to characteristic failed")
            } else {
                // When a 'JumpToBootloader', 'Activate and Reset' or 'Reset'
                // command is sent the device may reset before sending the acknowledgement.
                // This is not a blocker, as the device did disconnect and reset successfully.
                logger.a("\(request.description) request sent")
                logger.w("Device disconnected before sending ACK")
                logger.w(error)
                success?()
            }
            return
        }
        
        logger.i("Data written to \(characteristic.uuid.uuidString)")
        
        switch request {
        case .startDfu(_), .startDfu_v1,  .validateFirmware:
            logger.a("\(request.description) request sent")
            // do not call success until we get a notification
        case .jumpToBootloader, .activateAndReset, .reset, .packetReceiptNotificationRequest(_):
            logger.a("\(request.description) request sent")
            // there will be no notification send after these requests, call
            // `success()` immediatelly (for `.receiveFirmwareImage` the notification
            // will be sent after firmware upload is complete)
            success?()
        case .initDfuParameters(_), .initDfuParameters_v1:
            // Log was created before sending the Op Code.
            // Do not call success until we get a notification.
            break
        case .receiveFirmwareImage:
            if proceed == nil {
                success?()
            }
        }
    }
    
    func peripheral(_ peripheral: CBPeripheral,
                    didUpdateValueFor characteristic: CBCharacteristic,
                    error: Error?) {
        // Ignore updates received for other characteristics.
        guard self.characteristic.isEqual(characteristic) else {
            return
        }

        if let error = error {
            // This characteristic is never read, the error may only pop up when
            // notification is received.
            logger.e("Receiving notification failed")
            logger.e(error)
            report?(.receivingNotificationFailed, "Receiving notification failed")
            return
        }
        
        guard let characteristicValue = characteristic.value else { return }
        
        // During the upload we may get either a Packet Receipt Notification,
        // or a Response with status code.
        if proceed != nil {
            if let prn = PacketReceiptNotification(characteristicValue) {
                proceed!(prn.bytesReceived)
                return
            }
        }
        // Otherwise...
        logger.i("Notification received from \(characteristic.uuid.uuidString), value (0x): \(characteristicValue.hexString)")
        
        // Parse response received.
        let response = Response(characteristicValue)
        if let response = response {
            logger.a("\(response.description) received")
            
            if response.status == .success {
                switch response.requestOpCode {
                case .initDfuParameters:
                    logger.a("Initialize DFU Parameters completed")
                case .receiveFirmwareImage:
                    let interval = CFAbsoluteTimeGetCurrent() - uploadStartTime! as CFTimeInterval
                    logger.a("Upload completed in \(interval.format(".2")) seconds")
                default:
                    break
                }
                success?()
            } else {
                logger.e("Error \(response.status.code): \(response.status.description)")
                report?(response.status.error, response.status.description)
            }
        } else {
            logger.e("Unknown response received: 0x\(characteristicValue.hexString)")
            report?(.unsupportedResponse, "Unsupported response received: 0x\(characteristicValue.hexString)")
        }
    }
    
    func peripheralIsReady(toSendWriteWithoutResponse peripheral: CBPeripheral) {
        // On iOS 11 and MacOS 10.13 or newer PRS are no longer required. Instead,
        // the service checks if it can write write without response before writing
        // and it will get this callback if the buffer is ready again.
        proceed?(nil) // no offset available
    }
}
