/*
* Copyright (c) 2019, Nordic Semiconductor
* All rights reserved.
*
* Redistribution and use in source and binary forms, with or without modification,
* are permitted provided that the following conditions are met:
*
* 1. Redistributions of source code must retain the above copyright notice, this
*    list of conditions and the following disclaimer.
*
* 2. Redistributions in binary form must reproduce the above copyright notice, this
*    list of conditions and the following disclaimer in the documentation and/or
*    other materials provided with the distribution.
*
* 3. Neither the name of the copyright holder nor the names of its contributors may
*    be used to endorse or promote products derived from this software without
*    specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
* ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
* IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
* INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
* NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
* PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
* WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
* ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
* POSSIBILITY OF SUCH DAMAGE.
*/

import CoreBluetooth

internal typealias VersionCallback = (_ major: UInt8, _ minor: UInt8) -> Void

@objc internal class DFUVersion : NSObject, CBPeripheralDelegate, DFUCharacteristic {
    
    internal var characteristic: CBCharacteristic
    internal var logger: LoggerHelper

    private var success: VersionCallback?
    private var report: ErrorCallback?
    
    internal var valid: Bool {
        return characteristic.properties.contains(.read)
    }
    
    // MARK: - Initialization
    
    required init(_ characteristic: CBCharacteristic, _ logger: LoggerHelper) {
        self.characteristic = characteristic
        self.logger = logger
    }
    
    // MARK: - Characteristic API methods
    
    /**
     Reads the value of the DFU Version characteristic.
     The value, or an error, will be reported as a callback.
    
     - parameter success: Method called when version is read and is supported.
     - parameter report:  Method called on error of if version is not supported.
     */
    func readVersion(onSuccess success: VersionCallback?, onError report: ErrorCallback?) {
        // Get the peripheral object.
        let optService: CBService? = characteristic.service
        guard let peripheral = optService?.peripheral else {
            report?(.invalidInternalState, "Assert characteristic.service?.peripheral != nil failed")
            return
        }
        
        // Save callbacks.
        self.success = success
        self.report = report
        
        // Set the peripheral delegate to self.
        peripheral.delegate = self
        
        logger.v("Reading DFU Version number...")
        logger.d("peripheral.readValue(\(characteristic.uuid.uuidString))")
        peripheral.readValue(for: characteristic)
    }
    
    // MARK: - Peripheral Delegate callbacks
    
    func peripheral(_ peripheral: CBPeripheral,
                    didUpdateValueFor characteristic: CBCharacteristic,
                    error: Error?) {
        // Ignore updates received for other characteristics.
        guard self.characteristic.isEqual(characteristic) else {
            return
        }

        if let error = error {
            logger.e("Reading DFU Version characteristic failed")
            logger.e(error)
            report?(.readingVersionFailed, "Reading DFU Version characteristic failed")
            return
        }
        
        let data = characteristic.value
        logger.i("Read Response received from \(characteristic.uuid.uuidString), value\(data != nil && data!.count > 0 ? " (0x): " + data!.hexString : ": 0 bytes")")
        
        // Validate data length
        if data?.count != 2 {
            logger.w("Invalid value: 2 bytes expected")
            report?(.readingVersionFailed, "Unsupported DFU Version: \(data != nil && data!.count > 0 ? "0x" + data!.hexString : "no value")")
            return
        }
        
        // Read major and minor
        let minor: UInt8 = data![0]
        let major: UInt8 = data![1]
        
        logger.a("Version number read: \(major).\(minor)")
        success?(major, minor)
    }
}
